// +build integration

package storage_test

import (
	"context"
	"strconv"
	"testing"

	"github.com/stretchr/testify/require"
)

func TestAddedList(t *testing.T) {
	testEnv := newTestEnv(t)
	storageClient := testEnv.storageClient

	t.Run("getting an added list that doesn't exist", func(t *testing.T) {
		testEnv.reset(t)

		dbAddedChannels, err := storageClient.GetAddedList(context.Background(), "1")
		require.NoError(t, err)
		require.Empty(t, dbAddedChannels)
	})

	t.Run("adding to added list for first time", func(t *testing.T) {
		testEnv.reset(t)

		err := storageClient.AddToAddedList(context.Background(), "1", "2", 0)
		require.NoError(t, err)

		dbAddedChannels, err := storageClient.GetAddedList(context.Background(), "1")

		require.NoError(t, err)
		require.Len(t, dbAddedChannels, 1)
		require.Equal(t, dbAddedChannels[0].ID, "2")
	})

	t.Run("adding to existing added list", func(t *testing.T) {
		testEnv.reset(t)

		err := storageClient.AddToAddedList(context.Background(), "1", "2", 0)
		require.NoError(t, err)

		err = storageClient.AddToAddedList(context.Background(), "1", "3", 0)
		require.NoError(t, err)

		dbAddedChannels, err := storageClient.GetAddedList(context.Background(), "1")
		require.NoError(t, err)

		require.Len(t, dbAddedChannels, 2)
		require.Equal(t, dbAddedChannels[0].ID, "2")
		require.Equal(t, dbAddedChannels[1].ID, "3")
	})

	t.Run("removing from empty added list", func(t *testing.T) {
		testEnv.reset(t)
		err := storageClient.DeleteFromAddedList(context.Background(), "1", "2")
		require.NoError(t, err)

		dbAddedChannels, err := storageClient.GetAddedList(context.Background(), "1")
		require.NoError(t, err)

		require.Len(t, dbAddedChannels, 0)
	})

	t.Run("removing from existing added list", func(t *testing.T) {
		testEnv.reset(t)
		err := storageClient.AddToAddedList(context.Background(), "1", "2", 0)
		require.NoError(t, err)

		err = storageClient.DeleteFromAddedList(context.Background(), "1", "2")
		require.NoError(t, err)

		dbAddedChannels, err := storageClient.GetAddedList(context.Background(), "1")
		require.NoError(t, err)

		require.Len(t, dbAddedChannels, 0)
	})

	t.Run("trying to add to added list at max capacity", func(t *testing.T) {
		testEnv.reset(t)
		for i := 0; i < 100; i++ {
			err := storageClient.AddToAddedList(context.Background(), "1", strconv.Itoa(i), 0)
			require.NoError(t, err)
		}

		err := storageClient.AddToAddedList(context.Background(), "1", "10000", 0)
		require.NoError(t, err)

		dbAddedChannels, err := storageClient.GetAddedList(context.Background(), "1")
		require.NoError(t, err)

		require.Len(t, dbAddedChannels, 100)
	})

	t.Run("removing an added-list", func(t *testing.T) {
		testEnv.reset(t)

		targetUserID := "11111111"
		err := storageClient.AddToAddedList(context.Background(), targetUserID, "22222222", 0)
		require.NoError(t, err)

		err = storageClient.DeleteAddedList(context.Background(), targetUserID)
		require.NoError(t, err)

		dbAddedChannels, err := storageClient.GetAddedList(context.Background(), targetUserID)
		require.NoError(t, err)
		require.Empty(t, dbAddedChannels)
	})
}
