// +build integration

package storage_test

import (
	"context"
	"testing"

	"github.com/stretchr/testify/require"
)

func TestGetList(t *testing.T) {
	testEnv := newTestEnv(t)
	testEnv.reset(t)
	storageClient := testEnv.storageClient
	userID := "111111111"

	list := []string{"222222222", "333333333", "444444444"}
	_, err := storageClient.SetList(context.Background(), userID, list)
	require.NoError(t, err)

	fetchedList, err := storageClient.GetList(context.Background(), userID)
	require.NoError(t, err)
	require.Equal(t, list, fetchedList)
}

func TestGetList_ItemNotFound(t *testing.T) {
	testEnv := newTestEnv(t)
	testEnv.reset(t)
	storageClient := testEnv.storageClient
	userID := "111111111"

	list, err := storageClient.GetList(context.Background(), userID)
	require.NoError(t, err)
	require.Nil(t, list)
}

func TestSetList_CreateList(t *testing.T) {
	testEnv := newTestEnv(t)
	testEnv.reset(t)
	storageClient := testEnv.storageClient
	userID := "111111111"

	list := []string{"222222222", "333333333", "444444444"}
	returnedList, err := storageClient.SetList(context.Background(), userID, list)
	require.NoError(t, err)
	require.Equal(t, list, returnedList)
}

func TestSetList_SetToEmptyList(t *testing.T) {
	testEnv := newTestEnv(t)
	testEnv.reset(t)
	storageClient := testEnv.storageClient
	userID := "111111111"

	list := make([]string, 0)
	returnedList, err := storageClient.SetList(context.Background(), userID, list)
	require.NoError(t, err)
	require.Nil(t, returnedList)
}

func TestSetList_UpdateList(t *testing.T) {
	testEnv := newTestEnv(t)
	testEnv.reset(t)
	storageClient := testEnv.storageClient
	userID := "111111111"

	_, err := storageClient.SetList(context.Background(), userID, []string{
		"222222222", "333333333", "444444444",
	})
	require.NoError(t, err)

	updatedList := []string{"222222222", "444444444"}
	returnedList, err := storageClient.SetList(context.Background(), userID, updatedList)
	require.NoError(t, err)
	require.Equal(t, updatedList, returnedList)
}

func TestSetList_RemoveDuplicates(t *testing.T) {
	testEnv := newTestEnv(t)
	testEnv.reset(t)
	storageClient := testEnv.storageClient
	userID := "111111111"

	// Try setting a list with duplicated channels.
	returnedList, err := storageClient.SetList(context.Background(), userID, []string{
		"222222222", "333333333", "444444444", "333333333",
	})
	require.NoError(t, err)
	expectedList := []string{"222222222", "333333333", "444444444"}
	require.Equal(t, expectedList, returnedList)
}
