// +build integration

package storage_test

import (
	"context"
	"testing"

	"code.justin.tv/live/autohost/lib"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/stretchr/testify/require"
)

func TestUpdateSettings_CreateNewItem(t *testing.T) {
	testEnv := newTestEnv(t)
	testEnv.reset(t)
	storageClient := testEnv.storageClient
	userID := "111111111"

	autohostStrategyOrdered := lib.AutohostStrategyOrdered
	updatedSettings, err := storageClient.UpdateSettings(context.Background(), userID, &lib.UpdateSettingsInput{
		Enabled:             aws.Bool(true),
		TeamHost:            aws.Bool(true),
		Strategy:            &autohostStrategyOrdered,
		DeprioritizeVodcast: aws.Bool(true),
	})
	require.NoError(t, err)

	expectedSettings := &lib.Settings{
		Enabled:             true,
		TeamHost:            true,
		Strategy:            lib.AutohostStrategyOrdered,
		DeprioritizeVodcast: true,
	}
	require.Equal(t, expectedSettings, updatedSettings)
}

func TestUpdateSettings_UpdateExistingItem(t *testing.T) {
	testEnv := newTestEnv(t)
	testEnv.reset(t)
	storageClient := testEnv.storageClient
	userID := "111111111"

	// Create an item.
	autohostStrategyOrdered := lib.AutohostStrategyOrdered
	_, err := storageClient.UpdateSettings(context.Background(), userID, &lib.UpdateSettingsInput{
		Enabled:             aws.Bool(true),
		TeamHost:            aws.Bool(true),
		Strategy:            &autohostStrategyOrdered,
		DeprioritizeVodcast: aws.Bool(true),
	})
	require.NoError(t, err)

	// Update one attribute.
	updatedSettings, err := storageClient.UpdateSettings(context.Background(), userID, &lib.UpdateSettingsInput{
		Enabled: aws.Bool(false),
	})
	require.NoError(t, err)

	// Verify that the attribute has been updated.
	expectedSettings := &lib.Settings{
		Enabled:             false,
		TeamHost:            true,
		Strategy:            lib.AutohostStrategyOrdered,
		DeprioritizeVodcast: true,
	}
	require.Equal(t, expectedSettings, updatedSettings)
}

func TestUpdateSettings_NoChanges(t *testing.T) {
	testEnv := newTestEnv(t)
	testEnv.reset(t)
	storageClient := testEnv.storageClient
	userID := "111111111"

	// Create an item.
	autohostStrategyOrdered := lib.AutohostStrategyOrdered
	_, err := storageClient.UpdateSettings(context.Background(), userID, &lib.UpdateSettingsInput{
		Enabled:             aws.Bool(true),
		TeamHost:            aws.Bool(true),
		Strategy:            &autohostStrategyOrdered,
		DeprioritizeVodcast: aws.Bool(true),
	})
	require.NoError(t, err)

	// Try to update the item with no changes..
	updatedSettings, err := storageClient.UpdateSettings(context.Background(), userID, &lib.UpdateSettingsInput{})
	require.NoError(t, err)

	// Verify that the attributes were not changed.
	expectedSettings := &lib.Settings{
		Enabled:             true,
		TeamHost:            true,
		Strategy:            lib.AutohostStrategyOrdered,
		DeprioritizeVodcast: true,
	}
	require.Equal(t, expectedSettings, updatedSettings)
}

func TestGetSettings(t *testing.T) {
	testEnv := newTestEnv(t)
	testEnv.reset(t)
	storageClient := testEnv.storageClient
	userID := "111111111"

	// Set a user's settings.
	autohostStrategyOrdered := lib.AutohostStrategyOrdered
	_, err := storageClient.UpdateSettings(context.Background(), userID, &lib.UpdateSettingsInput{
		Enabled:             aws.Bool(true),
		TeamHost:            aws.Bool(true),
		Strategy:            &autohostStrategyOrdered,
		DeprioritizeVodcast: aws.Bool(true),
	})
	require.NoError(t, err)

	// Fetch the user's settings.
	settings, err := storageClient.GetSettings(context.Background(), userID)
	require.NoError(t, err)

	// Verify that their values are correct.
	expectedSettings := &lib.Settings{
		Enabled:             true,
		TeamHost:            true,
		Strategy:            lib.AutohostStrategyOrdered,
		DeprioritizeVodcast: true,
	}
	require.Equal(t, expectedSettings, settings)
}

func TestGetSettings_ItemNotFound(t *testing.T) {
	testEnv := newTestEnv(t)
	testEnv.reset(t)
	storageClient := testEnv.storageClient
	userID := "111111111"

	settings, err := storageClient.GetSettings(context.Background(), userID)
	require.NoError(t, err)
	require.Nil(t, settings)
}

func TestDeleteSettings(t *testing.T) {
	testEnv := newTestEnv(t)
	testEnv.reset(t)
	storageClient := testEnv.storageClient
	userID := "111111111"

	// Create an item.
	_, err := storageClient.UpdateSettings(context.Background(), userID, &lib.UpdateSettingsInput{
		Enabled: aws.Bool(true),
	})
	require.NoError(t, err)

	// Delete the item
	err = storageClient.DeleteSettings(context.Background(), userID)
	require.NoError(t, err)

	// Verify that the attribute has been updated.
	settings, err := storageClient.GetSettings(context.Background(), userID)
	require.NoError(t, err)
	require.Nil(t, settings)
}
