package config

import (
	"os"

	"code.justin.tv/live/autohost/internal/localdynamo"

	"code.justin.tv/creator-collab/log/errors"
)

const (
	// EnvDev is the "development environment".  A service is running in the development environment
	// when it is running locally on someone's laptop.
	EnvDev = "development"
	// EnvCI is the "continuous integration environment".  This means the service is running in Jenkins.
	EnvCI = "ci"
	// EnvStaging is the staging environment.
	EnvStaging = "staging"
	// EnvProduction is the production environment.
	EnvProduction = "production"
)

type Config struct {
	DynamicConfigKeyPrefix  string
	EnableCloudWatchMetrics bool
	Environment             string
	RunningInEC2            bool
	ServerAddress           string
	UseDevelopmentLogger    bool
	ClientURL               ClientURL
	SecretManagerARN        SecretManagerARN
	DynamoDB                DynamoDB
}

type ClientURL struct {
	Clue      string
	Multiplex string
	Users     string
}

type SecretManagerARN struct {
	RollbarToken string
}

type DynamoDB struct {
	HostTableName       string
	LocalDynamo         bool
	LocalDynamoEndpoint string
	SettingsTableName   string
}

func GetConfig() (*Config, error) {
	env, set := os.LookupEnv("ENVIRONMENT")
	if !set {
		return nil, errors.New("Environment variable named \"ENVIRONMENT\" was not set")
	}

	switch env {
	case EnvDev:
		return &DevelopmentConfig, nil
	case EnvCI:
		return &CIConfig, nil
	case EnvStaging:
		return &StagingConfig, nil
	case EnvProduction:
		return &ProductionConfig, nil
	}

	return nil, errors.New("could not config for given environment", errors.Fields{
		"ENVIRONMENT": env,
	})
}

var DevelopmentConfig = Config{
	DynamicConfigKeyPrefix:  "development",
	EnableCloudWatchMetrics: false,
	Environment:             EnvDev,
	RunningInEC2:            false,
	ServerAddress:           ":8000",
	UseDevelopmentLogger:    true,
	ClientURL: ClientURL{
		Clue:      "https://staging.clue.twitch.a2z.com",
		Multiplex: "https://staging.multiplex.twitch.a2z.com",
		Users:     "https://dev.users-service.twitch.a2z.com",
	},
	DynamoDB: DynamoDB{
		HostTableName:       localdynamo.TableHost,
		LocalDynamo:         true,
		LocalDynamoEndpoint: "http://localhost:8401",
		SettingsTableName:   localdynamo.TableAutohostSettings,
	},
	SecretManagerARN: SecretManagerARN{
		RollbarToken: "",
	},
}

var CIConfig = Config{
	DynamicConfigKeyPrefix:  "ci",
	EnableCloudWatchMetrics: false,
	Environment:             EnvCI,
	RunningInEC2:            false,
	ServerAddress:           ":8000",
	UseDevelopmentLogger:    true,
	ClientURL: ClientURL{
		Clue:      "https://staging.clue.twitch.a2z.com",
		Multiplex: "https://staging.multiplex.twitch.a2z.com",
		Users:     "https://dev.users-service.twitch.a2z.com",
	},
	DynamoDB: DynamoDB{
		HostTableName:       localdynamo.TableHost,
		LocalDynamo:         true,
		LocalDynamoEndpoint: "http://db:8000",
		SettingsTableName:   localdynamo.TableAutohostSettings,
	},
	SecretManagerARN: SecretManagerARN{
		RollbarToken: "",
	},
}

var StagingConfig = Config{
	DynamicConfigKeyPrefix:  "staging",
	EnableCloudWatchMetrics: true,
	Environment:             EnvStaging,
	RunningInEC2:            true,
	ServerAddress:           ":8000",
	UseDevelopmentLogger:    false,
	ClientURL: ClientURL{
		Clue:      "https://staging.clue.twitch.a2z.com",
		Multiplex: "https://staging.multiplex.twitch.a2z.com",
		Users:     "https://dev.users-service.twitch.a2z.com",
	},
	DynamoDB: DynamoDB{
		HostTableName:       "Host-staging",
		LocalDynamo:         false,
		LocalDynamoEndpoint: "",
		SettingsTableName:   "staging-settings",
	},
	SecretManagerARN: SecretManagerARN{
		RollbarToken: "arn:aws:secretsmanager:us-west-2:038504843107:secret:AutohostWorker-Staging-Rollbar-gWH6KK",
	},
}

var ProductionConfig = Config{
	DynamicConfigKeyPrefix:  "production",
	EnableCloudWatchMetrics: true,
	Environment:             EnvProduction,
	RunningInEC2:            true,
	ServerAddress:           ":8000",
	UseDevelopmentLogger:    false,
	ClientURL: ClientURL{
		Clue:      "https://prod.clue.twitch.a2z.com",
		Multiplex: "https://production-internal.multiplex.twitch.a2z.com",
		Users:     "https://prod.users-service.twitch.a2z.com",
	},
	DynamoDB: DynamoDB{
		HostTableName:       "Host-production",
		LocalDynamo:         false,
		LocalDynamoEndpoint: "",
		SettingsTableName:   "production-settings",
	},
	SecretManagerARN: SecretManagerARN{
		RollbarToken: "arn:aws:secretsmanager:us-west-2:447680546588:secret:AutohostWorker-Production-Rollbar-eysN50",
	},
}
