package config

import (
	"code.justin.tv/creator-collab/log/errors"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/secretsmanager"
)

// Secrets contains the credentials that this service needs.
type Secrets struct {
	RollbarToken string
}

// LoadSecrets fetches this service's secrets from Secrets Manager.
func LoadSecrets(config *Config, awsSession *session.Session) (*Secrets, error) {
	secretsManagerClient := secretsmanager.New(awsSession)

	// We only have 1 secret - a Rollbar token. If we're not going to use the token, don't bother
	// querying secrets manager.
	if config.UseDevelopmentLogger {
		return &Secrets{}, nil
	}

	rollbarArn := config.SecretManagerARN.RollbarToken
	out, err := secretsManagerClient.GetSecretValue(&secretsmanager.GetSecretValueInput{
		SecretId: aws.String(rollbarArn),
	})
	if err != nil {
		return nil, errors.Wrap(err, "loading secret failed", errors.Fields{
			"secret_arn": rollbarArn,
		})
	}
	if out == nil || out.SecretString == nil || *out.SecretString == "" {
		return nil, errors.New("secret string is nil or empty", errors.Fields{
			"secret_arn": rollbarArn,
		})
	}

	return &Secrets{
		RollbarToken: *out.SecretString,
	}, nil
}
