package parser

import (
	"testing"

	"github.com/stretchr/testify/require"
)

// ParseBrazilPath tests

func TestParseBrazilPath_MainPackage(t *testing.T) {
	fullPath := "/local/p4clients/pkgbuild-XOA_p/workspace/src/StreamSchedule/storage/segments.go"
	expectedBrazilPath := &BrazilPath{
		Package:           "StreamSchedule",
		FilePath:          "storage/segments.go",
		HasMainlineBranch: true,
		IsServicePackage:  true,
	}
	brazilPath := ParseBrazilPath(fullPath)
	require.Equal(t, expectedBrazilPath, brazilPath)
}

func TestParseBrazilPath_Golang(t *testing.T) {
	fullPath := "/opt/brazil-pkg-cache/packages/GoLang/GoLang-1.0.284115.0/RHEL5_64/DEV.STD.PTHREAD/build/lib/src/runtime/asm_amd64.s"
	expectedBrazilPath := &BrazilPath{
		Package:  "GoLang",
		FilePath: "runtime/asm_amd64.s",
	}
	brazilPath := ParseBrazilPath(fullPath)
	require.Equal(t, expectedBrazilPath, brazilPath)
}

func TestParseBrazilPath_GeneratedTwirp(t *testing.T) {
	fullPath := "/local/p4clients/pkgbuild-xbNT4/workspace/src/TwitchNexus/build/private/bgospace/TwitchNexusTwirp/src/code.justin.tv/amzn/TwitchNexusTwirp/twirp.twirp.go"
	expectedBrazilPath := &BrazilPath{
		Package:  "TwitchNexusTwirp",
		FilePath: "twirp.twirp.go",
	}
	brazilPath := ParseBrazilPath(fullPath)
	require.Equal(t, expectedBrazilPath, brazilPath)
}

func TestParseBrazilPath_ThirdParty_Workspace(t *testing.T) {
	fullPath := "/local/p4clients/pkgbuild-xbNT4/workspace/src/TwitchNexus/build/private/bgospace/Go3p-Github-Twitchtv-Twirp/src/github.com/twitchtv/twirp/hooks.go"
	expectedBrazilPath := &BrazilPath{
		Package:  "Go3p-Github-Twitchtv-Twirp",
		FilePath: "hooks.go",
	}
	brazilPath := ParseBrazilPath(fullPath)
	require.Equal(t, expectedBrazilPath, brazilPath)
}

func TestParseBrazilPath_ThirdParty_BrazilCache(t *testing.T) {
	fullPath := "/opt/brazil-pkg-cache/packages/StreamSchedule/StreamSchedule-1.0.1215.0/RHEL5_64/generic-flavor/src/build/private/bgospace/Go3p-Github-Twitchtv-Twirp/src/github.com/twitchtv/twirp/hooks.go"
	expectedBrazilPath := &BrazilPath{
		Package:  "Go3p-Github-Twitchtv-Twirp",
		FilePath: "hooks.go",
	}
	brazilPath := ParseBrazilPath(fullPath)
	require.Equal(t, expectedBrazilPath, brazilPath)
}

func TestParseBrazilPath_VXPGolibs_BrazilCache(t *testing.T) {
	fullPath := "/opt/brazil-pkg-cache/packages/StreamSchedule/StreamSchedule-1.0.1215.0/RHEL5_64/generic-flavor/src/build/private/bgospace/Twitch-VXP-Golibs/src/golang.a2z.com/Twitch-VXP-Golibs/middleware/logger.go"
	expectedBrazilPath := &BrazilPath{
		Package:           "Twitch-VXP-Golibs",
		FilePath:          "middleware/logger.go",
		HasMainlineBranch: true,
	}
	brazilPath := ParseBrazilPath(fullPath)
	require.Equal(t, expectedBrazilPath, brazilPath)
}

// tryParseDependency tests

func TestTryParseDependency_Success_RepeatedPackage(t *testing.T) {
	brazilPkg := tryParseDependency("/local/p4clients/pkgbuild-tG5qG/workspace/src/StreamSchedule/build/private/bgospace/StreamScheduleTwirp/src/code.justin.tv/amzn/StreamScheduleTwirp/twirp.twirp.go")
	require.NotNil(t, brazilPkg)
	require.Equal(t, "StreamScheduleTwirp", brazilPkg.Package)
	require.Equal(t, "twirp.twirp.go", brazilPkg.FilePath)
	require.False(t, brazilPkg.IsServicePackage)
	require.False(t, brazilPkg.HasMainlineBranch)
}

func TestTryParseDependency_Success_Go3p(t *testing.T) {
	brazilPkg := tryParseDependency("/local/p4clients/pkgbuild-xbNT4/workspace/src/TwitchNexus/build/private/bgospace/Go3p-Github-Twitchtv-Twirp/src/github.com/twitchtv/twirp/hooks.go")
	require.NotNil(t, brazilPkg)
	require.Equal(t, "Go3p-Github-Twitchtv-Twirp", brazilPkg.Package)
	require.Equal(t, "hooks.go", brazilPkg.FilePath)
	require.False(t, brazilPkg.IsServicePackage)
	require.False(t, brazilPkg.HasMainlineBranch)
}

func TestTryParseDependency_Success_RepeatedPackageInBrazilCache(t *testing.T) {
	brazilPkg := tryParseDependency("/opt/brazil-pkg-cache/packages/StreamSchedule/StreamSchedule-1.0.1215.0/RHEL5_64/generic-flavor/src/build/private/bgospace/Twitch-VXP-Golibs/src/golang.a2z.com/Twitch-VXP-Golibs/middleware/logger.go")
	require.NotNil(t, brazilPkg)
	require.Equal(t, "Twitch-VXP-Golibs", brazilPkg.Package)
	require.Equal(t, "middleware/logger.go", brazilPkg.FilePath)
	require.False(t, brazilPkg.IsServicePackage)
	require.True(t, brazilPkg.HasMainlineBranch)
}

func TestTryParseDependency_Fail(t *testing.T) {
	brazilPkg := tryParseDependency("/local/p4clients/pkgbuild-XOA_p/workspace/src/StreamSchedule/storage/segments.go")
	require.Nil(t, brazilPkg)
}

// shortenGo3pDependencyPath tests

func TestShortenGo3pDependencyPath_Success(t *testing.T) {
	path := shortenGo3pDependencyPath("Go3p-Github-Twitchtv-Twirp", "github.com/twitchtv/twirp/hooks.go")
	require.Equal(t, "hooks.go", path)
}

func TestShortenGo3pDependencyPath_Fail(t *testing.T) {
	longPathThatDoesntMatchPattern := "github.com/a/twirp/hooks.go"
	path := shortenGo3pDependencyPath("Go3p-Github-Twitchtv-Twirp", longPathThatDoesntMatchPattern)
	require.Equal(t, longPathThatDoesntMatchPattern, path)
}
