package server

import (
	"code.justin.tv/creator-collab/log/errors"
)

const (
	ServiceIDAutohost       = "autohost"
	ServiceIDNexus          = "nexus"
	ServiceIDRaids          = "raids"
	ServiceIDShelfie        = "shelfie"
	ServiceIDStreamSchedule = "streamschedule"
)

const (
	StageProd = "prod"
	StageBeta = "beta"
)

type AWSAccount struct {
	ID   string
	Name string
	Role string
}

var AwsAccountRaidsProd = &AWSAccount{
	ID:   "947153321414",
	Name: "TwitchRaids-us-west-2",
	Role: "LowRiskReadOnly",
}

var AwsAccountNexusProd = &AWSAccount{
	ID:   "981671725203",
	Name: "twitch-nexus",
	Role: "LowRiskReadOnly",
}

var AwsAccountStreamScheduleProd = &AWSAccount{
	ID:   "524336105177",
	Name: "StreamSchedule-us-west-2",
	Role: "LowRiskReadOnly",
}

type Service struct {
	ID           string
	StageID      string
	DisplayName  string
	DisplayStage string
	URLSlug      string
	LogGroups    []string
}

var ServiceStreamScheduleProd = &Service{
	ID:           ServiceIDStreamSchedule,
	StageID:      StageProd,
	DisplayName:  "Stream Schedule",
	DisplayStage: "Prod",
	URLSlug:      "streamschedule/prod",
	LogGroups: []string{
		"/aws/ecs/StreamSchedule-prod",
		"/aws/ecs/StreamSchedule-prod-Canary",
	},
}

var ServiceStreamScheduleBeta = &Service{
	ID:           ServiceIDStreamSchedule,
	StageID:      StageBeta,
	DisplayName:  "Stream Schedule",
	DisplayStage: "Beta",
	URLSlug:      "streamschedule/beta",
	LogGroups: []string{
		"/aws/ecs/StreamSchedule-beta",
		"/aws/ecs/StreamSchedule-beta-Canary",
	},
}

var ServiceNexusProd = &Service{
	ID:           ServiceIDNexus,
	StageID:      StageProd,
	DisplayName:  "Nexus",
	DisplayStage: "Prod",
	URLSlug:      "nexus/prod",
	LogGroups: []string{
		"/aws/ecs/TwitchNexus-prod",
		"/aws/ecs/TwitchNexus-prod-Canary",
	},
}

var ServiceNexusBeta = &Service{
	ID:           ServiceIDNexus,
	StageID:      StageBeta,
	DisplayName:  "Nexus",
	DisplayStage: "Beta",
	URLSlug:      "nexus/beta",
	LogGroups: []string{
		"/aws/ecs/TwitchNexus-beta",
		"/aws/ecs/TwitchNexus-beta-Canary",
	},
}

var ServiceRaidsProd = &Service{
	ID:           ServiceIDRaids,
	StageID:      StageProd,
	DisplayName:  "Raids",
	DisplayStage: "Prod",
	URLSlug:      "raids/prod",
	LogGroups: []string{
		"/aws/ecs/TwitchRaids-prod",
		"/aws/ecs/TwitchRaids-prod-Canary",
	},
}

var ServiceRaidsBeta = &Service{
	ID:           ServiceIDRaids,
	StageID:      StageProd,
	DisplayName:  "Raids",
	DisplayStage: "Beta",
	URLSlug:      "raids/beta",
	LogGroups: []string{
		"/aws/ecs/TwitchRaids-beta",
		"/aws/ecs/TwitchRaids-beta-Canary",
	},
}

var ServiceShelfieProd = &Service{
	ID:           ServiceIDShelfie,
	StageID:      StageProd,
	DisplayName:  "Shelfie",
	DisplayStage: "Prod",
	URLSlug:      "shelfie/prod",
	LogGroups: []string{
		"/aws/ecs/Shelfie-prod",
		"/aws/ecs/Shelfie-prod-Canary",
	},
}

var ServiceShelfieBeta = &Service{
	ID:           ServiceIDShelfie,
	StageID:      StageBeta,
	DisplayName:  "Shelfie",
	DisplayStage: "Beta",
	URLSlug:      "shelfie/beta",
	LogGroups: []string{
		"/aws/ecs/Shelfie-beta",
		"/aws/ecs/Shelfie-beta-Canary",
	},
}

func GetService(serviceID string, stage string) (*Service, error) {
	switch serviceID {
	case ServiceIDStreamSchedule:
		switch stage {
		case StageProd:
			return ServiceStreamScheduleProd, nil
		case StageBeta:
			return ServiceStreamScheduleBeta, nil
		}

	case ServiceIDNexus:
		switch stage {
		case StageProd:
			return ServiceNexusProd, nil
		case StageBeta:
			return ServiceNexusBeta, nil
		}

	case ServiceIDRaids:
		switch stage {
		case StageProd:
			return ServiceRaidsProd, nil
		case StageBeta:
			return ServiceRaidsBeta, nil
		}

	case ServiceIDShelfie:
		switch stage {
		case StageProd:
			return ServiceShelfieProd, nil
		case StageBeta:
			return ServiceShelfieBeta, nil
		}
	}

	return nil, errors.New("no service for given service ID and stage", errors.Fields{
		"service_id": serviceID,
		"stage":      stage,
	})
}
