import { getPluckyClient } from "utils";
import useSWR from "swr";
import "./App.css";
import { Service } from "plucky_client/plucky";
import { ItemsPage } from "pages/items";
import { OccurrencesPage } from "pages/occurrences";
import { BrowserRouter as Router, Switch, Route, Link } from "react-router-dom";
import { CloudwatchLogsQueryLocationState } from "plucky_client/hooks";
import { useAppTimeRanges } from "hooks/use-timerange";

function App() {
  const appTimeRanges = useAppTimeRanges();

  const { data, error } = useSWR(
    "projects-query",
    (query) => {
      const pluckyClient = getPluckyClient();
      return pluckyClient.getConfig({});
    },
    {
      shouldRetryOnError: false,
      revalidateOnFocus: false,
    }
  );
  if (error) {
    return <div>failed to load services {error.toString()} </div>;
  }
  if (!data) {
    return <div>loading</div>;
  }

  const services = data.services;
  const serviceLinks = services.map((service: Service) => {
    return (
      <Link<CloudwatchLogsQueryLocationState>
        to={{
          pathname: `/${service.urlSlug}/items`,
          state: {
            skipPartialDataRefetch: true,
          },
        }}
        className="navlink"
        key={service.serviceId}
      >
        {service.displayName} ({service.displayStage})
      </Link>
    );
  });

  const serviceMap = new Map<string, Service>();
  for (let service of services) {
    serviceMap.set(service.serviceId + service.stageId, service);
  }

  return (
    <div className="app">
      <Router>
        <nav className="topnav">{serviceLinks}</nav>
        <main className="main">
          <Switch>
            <Route
              path="/:project/:stage/items"
              render={(routeProps) => {
                const project: string = routeProps.match.params.project;
                const stage: string = routeProps.match.params.stage;
                const service = serviceMap.get(project + stage);
                if (!service) {
                  return <div>Service not found</div>;
                }

                return (
                  <ItemsPage service={service} appTimeRanges={appTimeRanges} />
                );
              }}
            />
            <Route
              path="/:project/:stage/occurrences/:fingerprint"
              render={(routeProps) => {
                const project: string = routeProps.match.params.project;
                const stage: string = routeProps.match.params.stage;
                const service = serviceMap.get(project + stage);
                if (!service) {
                  return <div>Service not found</div>;
                }

                const fingerprint: string = routeProps.match.params.fingerprint;

                return (
                  <OccurrencesPage
                    service={service}
                    fingerprint={fingerprint}
                    appTimeRanges={appTimeRanges}
                  />
                );
              }}
            />
          </Switch>
        </main>
        <footer className="version">
          Version: {data.version}, Git Commit: {data.gitCommit}
        </footer>
      </Router>
    </div>
  );
}

export default App;
