import React, { useState } from "react";
import { Item as PluckyItem, Trend } from "plucky_client/plucky";
import styles from "./item_table.module.css";
import errorIconPath from "./error.svg";
import warningIconPath from "./warning.svg";
import panicIconPath from "./panic.svg";
import infoIconPath from "./info.svg";
import { TrendCell } from "components/trend_cell";
import { DateTime } from "luxon";
import { ExpandButton } from "components/expand_button";
import { OccurrenceDetails } from "components/occurrence_details";
import { Link as ReactRouterLink } from "react-router-dom";
import { QueryRange } from "utils";
import { Link } from "@chakra-ui/react";
import { linkColor } from "theme";

function getIconPath(level: string): string {
  switch (level) {
    case "panic":
      return panicIconPath;
    case "error":
      return errorIconPath;
    case "warning":
      return warningIconPath;
    default:
      return infoIconPath;
  }
}

function getTotal(trend: Trend): number {
  let total = 0;
  for (let point of trend.points) {
    total += point.count || 0;
  }
  return total;
}

export interface ItemRowProps {
  item: PluckyItem;
  queryRange: QueryRange;
  serviceURLSlug: string;
  showOperation: boolean;
}

export function ItemRow(props: ItemRowProps) {
  const item = props.item;
  const occurrence = item.lastOccurrence;
  const msg = occurrence.message;
  const level = occurrence.level;
  const fingerprint = item.fingerprint;
  const operation = occurrence.operation;

  const iconPath = getIconPath(level);

  let total = getTotal(item.trend);

  const dateTime = DateTime.fromJSDate(occurrence.timestamp);
  const last = dateTime.toLocaleString(DateTime.TIME_WITH_SHORT_OFFSET);
  const [showDetails, updateShowDetails] = useState(false);

  const operationRow = props.showOperation ? (
    <div className={styles.row}>{operation}</div>
  ) : null;

  let detailsRow = null;
  if (showDetails) {
    detailsRow = (
      <div className={styles.detailsRow}>
        <OccurrenceDetails
          occurrence={occurrence}
          headingText="Latest Occurrence"
        />
      </div>
    );
  }
  return (
    <>
      <div className={styles.row}>
        <img src={iconPath} alt={level} />
      </div>
      <div className={styles.row}>
        <Link
          as={ReactRouterLink}
          color={linkColor}
          to={{
            pathname: `/${props.serviceURLSlug}/occurrences/${fingerprint}`,
            state: {
              skipPartialDataRefetch: true,
            },
          }}
        >
          {msg}
        </Link>
      </div>
      <div className={styles.row}>
        <TrendCell trend={item.trend} queryRange={props.queryRange} />
      </div>
      <div className={`${styles.row} ${styles.totalCell}`}>{total}</div>
      <div className={styles.row}>{last}</div>
      {operationRow}
      <div className={styles.row}>{level}</div>
      <div className={styles.row}>
        <ExpandButton
          expanded={showDetails}
          onToggle={(expanded) => {
            updateShowDetails(expanded);
          }}
        />
      </div>
      <div className={styles.rowLine} />

      {detailsRow}
    </>
  );
}
