package youtube

import (
	"fmt"
	"testing"

	"github.com/stretchr/testify/assert"

	"google.golang.org/api/youtube/v3"
)

func TestShiftNonConflictingPromotion(t *testing.T) {
	items := []*youtube.PromotedItem{
		{
			Timing: &youtube.InvideoTiming{
				Type:       "offsetFromStart",
				OffsetMs:   15000,
				DurationMs: 2000,
			},
		},
	}
	shiftPromotions(items)
	assert.Equal(t, uint64(15000), items[0].Timing.OffsetMs)
	assert.Equal(t, uint64(2000), items[0].Timing.DurationMs)
}

func TestShiftOneConflictingPromotion(t *testing.T) {
	items := []*youtube.PromotedItem{
		{
			Timing: &youtube.InvideoTiming{
				Type:       "offsetFromStart",
				OffsetMs:   10000,
				DurationMs: 2000,
			},
		},
	}
	shiftPromotions(items)
	assert.Equal(t, uint64(14000), items[0].Timing.OffsetMs)
	assert.Equal(t, uint64(2000), items[0].Timing.DurationMs)
}

func TestShiftMultipleConflictingPromotion(t *testing.T) {
	items := []*youtube.PromotedItem{
		{
			Timing: &youtube.InvideoTiming{
				Type:       "offsetFromStart",
				OffsetMs:   10000,
				DurationMs: 2000,
			},
		},
		{
			Timing: &youtube.InvideoTiming{
				Type:       "offsetFromStart",
				OffsetMs:   5000,
				DurationMs: 4000,
			},
		},
	}
	shiftPromotions(items)
	fmt.Printf("%T", items[0].Timing.OffsetMs)
	assert.Equal(t, uint64(19000), items[0].Timing.OffsetMs)
	assert.Equal(t, uint64(2000), items[0].Timing.DurationMs)
	assert.Equal(t, uint64(14000), items[1].Timing.OffsetMs)
	assert.Equal(t, uint64(4000), items[1].Timing.DurationMs)
}

func TestIsTwitchPromotion(t *testing.T) {
	item := &youtube.PromotedItem{
		Id: &youtube.PromotedItemId{
			Type:       "website",
			WebsiteUrl: "https://www.twitch.tv/yt/asdfasdf",
		},
	}
	assert.Equal(t, true, isTwitchPromotion(item))
}

func TestIsNotTwitchWebsitePromotion(t *testing.T) {
	item := &youtube.PromotedItem{
		Id: &youtube.PromotedItemId{
			Type:       "website",
			WebsiteUrl: "https://www.someother.url",
		},
	}
	assert.Equal(t, false, isTwitchPromotion(item))
}

func TestIsNotTwitchVideoPromotion(t *testing.T) {
	item := &youtube.PromotedItem{
		Id: &youtube.PromotedItemId{
			Type: "video",
		},
	}
	assert.Equal(t, false, isTwitchPromotion(item))
}

func TestTwitchAnnotationNotEnabled(t *testing.T) {
	channel := &youtube.Channel{
		InvideoPromotion: &youtube.InvideoPromotion{
			Items: []*youtube.PromotedItem{
				{
					Id: &youtube.PromotedItemId{
						Type: "video",
					},
				},
				{
					Id: &youtube.PromotedItemId{
						Type:       "website",
						WebsiteUrl: "https://not.twitch.tv",
					},
				},
			},
		},
	}

	assert.Equal(t, false, twitchAnnotationsEnabled(channel))
}

func TestTwitchAnnotationEnabled(t *testing.T) {
	channel := &youtube.Channel{
		InvideoPromotion: &youtube.InvideoPromotion{
			Items: []*youtube.PromotedItem{
				{
					Id: &youtube.PromotedItemId{
						Type: "video",
					},
				},
				{
					Id: &youtube.PromotedItemId{
						Type:       "website",
						WebsiteUrl: "https://www.twitch.tv/yt/asdf",
					},
				},
			},
		},
	}

	assert.Equal(t, true, twitchAnnotationsEnabled(channel))
}
