package server

import (
	"log"
	"net/http"

	"code.justin.tv/marketing/altlookup/definitions"

	"golang.org/x/net/context"
)

const maxAltAccounts = 500

func (s *Server) GetUserAlts(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	userID := r.URL.Query().Get("user_id")
	resp, err := s.getUserAlts(ctx, userID)
	if err != nil {
		serveError(ctx, w, r, err, http.StatusInternalServerError)
	}

	serveJSON(w, r, resp)
}

func (s *Server) getUserAlts(ctx context.Context, userID string) ([]definitions.SafeAccountInfo, error) {
	totalAlternativeAccounts := []definitions.AccountInfo{}
	totalLookedUpCount := 0

	initialLookupAlternates, err := s.clients.SiteDB.AlternateAccountsOfUser(ctx, userID)
	if err != nil {
		return nil, err
	}
	for _, alt := range initialLookupAlternates {
		totalAlternativeAccounts = append(totalAlternativeAccounts, alt)
	}

	emailAltsOfInitial, err := s.clients.SiteDB.AlternateAccountsOfEmail(ctx, userID, totalAlternativeAccounts[0].Email)
	if err != nil {
		return nil, err
	}
	for _, alt := range emailAltsOfInitial {
		totalAlternativeAccounts = append(totalAlternativeAccounts, alt)
	}

	ipAltsOfInitial, err := s.clients.SiteDB.AlternateAccountsOfIP(ctx, userID, totalAlternativeAccounts[0].IP)
	if err != nil {
		return nil, err
	}
	for _, alt := range ipAltsOfInitial {
		totalAlternativeAccounts = append(totalAlternativeAccounts, alt)
	}

	totalAlternativeAccounts = removeDuplicateAlts(totalAlternativeAccounts)

	altsOfAlts := []definitions.AccountInfo{}
	for _, alt := range totalAlternativeAccounts {
		alts, err := s.getAltsOfAlt(ctx, alt)
		if err != nil {
			return nil, err
		}
		for _, alt := range alts {
			altsOfAlts = append(altsOfAlts, alt)
		}
		totalLookedUpCount++
		if totalLookedUpCount > maxAltAccounts {
			return altInfoMinusPersonal(totalAlternativeAccounts), nil
		}
	}
	for _, alt := range altsOfAlts {
		totalAlternativeAccounts = append(totalAlternativeAccounts, alt)
	}

	totalAlternativeAccounts = removeDuplicateAlts(totalAlternativeAccounts)

	safeAlts := altInfoMinusPersonal(totalAlternativeAccounts)

	log.Printf("Returning %d alternate accounts for user ID %s", len(safeAlts), userID)
	return safeAlts, nil

}

func altInfoMinusPersonal(alts []definitions.AccountInfo) []definitions.SafeAccountInfo {
	safeAltInfo := []definitions.SafeAccountInfo{}
	for _, alt := range alts {
		safeAlt := definitions.SafeAccountInfo{
			UserID: alt.UserID,
			Login:  alt.Login,
			TermsOfServiceViolation: alt.TermsOfServiceViolation,
			ParentAccountID:         alt.ParentAccountID,
			ParentAccountMatchType:  alt.ParentAccountMatchType,
			CreatedOn:               alt.CreatedOn,
		}
		safeAltInfo = append(safeAltInfo, safeAlt)
	}
	if len(safeAltInfo) > maxAltAccounts {
		safeAltInfo = safeAltInfo[:maxAltAccounts]
	}
	return safeAltInfo
}

func (s *Server) getAltsOfAlt(ctx context.Context, account definitions.AccountInfo) ([]definitions.AccountInfo, error) {
	alts := []definitions.AccountInfo{}
	emailAlts, err := s.clients.SiteDB.AlternateAccountsOfEmail(ctx, account.UserID, account.Email)
	if err != nil {
		return nil, err
	}
	for _, alt := range emailAlts {
		alts = append(alts, alt)
	}

	ipAlts, err := s.clients.SiteDB.AlternateAccountsOfIP(ctx, account.UserID, account.IP)
	if err != nil {
		return nil, err
	}
	for _, alt := range ipAlts {
		alts = append(alts, alt)
	}
	return alts, nil
}

func removeDuplicateAlts(elements []definitions.AccountInfo) []definitions.AccountInfo {
	newAlts := []definitions.AccountInfo{}
	for _, alt := range elements {
		found := false
		for _, newAlt := range newAlts {
			if alt.Login == newAlt.Login {
				found = true
			}
		}
		if !found {
			newAlts = append(newAlts, alt)
		}
	}
	return newAlts
}
