package gojiplus

import (
	"fmt"
	"log"
	"net/http"
	"time"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/zenazn/goji/graceful"
	"goji.io"

	"code.justin.tv/common/chitin"
	"code.justin.tv/common/gometrics"
	"code.justin.tv/common/meh-restart/restart"
)

var gometricsInterval = 5 * time.Second

// Decorator is used to provide re-usable decorators for goji.HandlerFuncs
type Decorator func(goji.HandlerFunc) goji.HandlerFunc

// ServerParams provide configuration options for servers
type ServerParams struct {
	Port      int
	DebugPort int
	Stats     statsd.Statter
}

// ListenAndServe starts an opinionated application server. Implements the following:
// - debug server: pprof debugging
// - go runtime metrics: system stats
// - chitin: request trace stats
// - graceful and meh-restart: graceful restarts
func ListenAndServe(mux http.Handler, params ServerParams) error {
	go func() {
		addr := fmt.Sprintf(":%d", params.DebugPort)
		log.Println("debug server listening on", addr)
		log.Println(http.ListenAndServe(addr, nil))
	}()

	gometrics.Monitor(params.Stats, gometricsInterval)

	var muxWithChitin http.Handler
	if err := chitin.ExperimentalTraceProcessOptIn(); err != nil {
		return err
	}
	if gojiMux, ok := mux.(goji.Handler); ok {
		muxWithChitin = chitin.Handler(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			gojiMux.ServeHTTPC(chitin.Context(w, r), w, r)
		}))
	} else {
		muxWithChitin = chitin.Handler(mux)
	}

	restart.OnShutdown(func() {
		graceful.Shutdown()
	})
	addr := fmt.Sprintf(":%d", params.Port)
	log.Println("application server listening on", addr)
	if err := graceful.ListenAndServe(addr, muxWithChitin); err != nil {
		return err
	}
	graceful.Wait()
	return nil
}

// HealthCheck is a "meh-restart"-compatible noop handler.
func HealthCheck() http.Handler {
	return restart.HealthCheck(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)
	}))
}
