package gojiplus

import (
	"net"
	"net/http"
	"strconv"
	"strings"
)

const (
	// AuthOAuth is the Authorization prefix for an OAuth token.
	AuthOAuth = "OAuth"

	// AuthBearer is the Authorization prefix for an shared secret.
	AuthBearer = "Bearer"
)

// GetJSONPCallback fetches the JSONP callback.
func GetJSONPCallback(r *http.Request) string {
	return r.FormValue("callback")
}

// ParseAuthorizationHeader parses the "Authorization" header for a scheme and token.
func ParseAuthorizationHeader(r *http.Request) (scheme, token string, ok bool) {
	header := fetchHeader(r, "Authorization")
	split := strings.SplitN(header, " ", 2)
	if len(split) != 2 {
		return "", "", false
	}
	// Support optional ":" in header (e.g. "OAuth: <token>")
	scheme = strings.TrimPrefix(split[0], ":")
	token = split[1]
	if scheme == "" || token == "" || (scheme != AuthOAuth && scheme != AuthBearer) {
		return "", "", false
	}
	return scheme, token, true
}

// HostPort combines a host and port to a hostport string.
// - "www.foobar.com" + 443		=>	"www.foobar.com:443"
// - "www.foobar.com" + 0		=>	"www.foobar.com"
func HostPort(host string, port int) string {
	if port == 0 {
		return host
	}
	return net.JoinHostPort(host, strconv.Itoa(port))
}

func fetchHeader(r *http.Request, name string) string {
	h := r.Header.Get(name)
	if h == "" {
		return ""
	}
	return strings.TrimSpace(strings.Split(h, ",")[0])
}
