package jsonconf

import (
	"errors"
	"fmt"
	"time"

	"gopkg.in/validator.v2"
)

var (
	jsonValidator = validator.NewValidator()
)

func init() {
	jsonValidator.SetTag("validate")
	SetValidationFunc("duration_min", validateDuration("duration_min", validateDurationMin))
	SetValidationFunc("duration_max", validateDuration("duration_max", validateDurationMax))
}

type ValidationFunc validator.ValidationFunc

func SetValidationFunc(name string, vf ValidationFunc) error {
	return jsonValidator.SetValidationFunc(name, validator.ValidationFunc(vf))
}

func validateDuration(key string, test func(d, cmp time.Duration, param string) error) ValidationFunc {
	return func(v interface{}, param string) error {
		cmp, err := time.ParseDuration(param)
		if err != nil {
			return fmt.Errorf("invalid %s=%s", key, param)
		}
		d, ok := v.(Duration)
		if !ok {
			return errors.New("must be jsonconf.Duration")
		}
		return test(time.Duration(d), cmp, param)
	}
}

func validateDurationMin(d, min time.Duration, param string) error {
	if d < min {
		return fmt.Errorf("duration less than %s", param)
	}
	return nil
}

func validateDurationMax(d, max time.Duration, param string) error {
	if d > max {
		return fmt.Errorf("duration greater than %s", param)
	}
	return nil
}
