package logx

import (
	"errors"
	"runtime"
	"sync"

	"github.com/Sirupsen/logrus"
	"golang.org/x/net/context"
)

type Fields logrus.Fields

// contextFields contains a mutable slice of fields
type contextFields struct {
	lock   sync.Mutex
	fields []Fields
}

var loggers = []Logger{}

func AddLogger(logger Logger) {
	loggers = append(loggers, logger)
}

// RecoverAndLog logs panics at error level. Use with defer:
// func main() {
//   defer logx.RecoverAndLog()
// }
func RecoverAndLog(size int) {
	if r := recover(); r != nil {
		buf := make([]byte, size)
		runtime.Stack(buf, true)
		Error(context.Background(), errors.New("panic recovered"), Fields{
			"stack": buf,
		})
	}
}

// Context initializes a new context to be used with logx.
func Context(ctx context.Context) context.Context {
	if ctx == nil {
		return ctx
	}
	return context.WithValue(ctx, contextFieldsKey, &contextFields{})
}

// WithFields returns a copy of the context with fields to log.
// Fields added to this context *will* be available to previous versions
// of the context.
func WithFields(ctx context.Context, fields Fields) context.Context {
	if ctx == nil {
		return ctx
	}

	cf, ok := ctx.Value(contextFieldsKey).(*contextFields)
	if !ok {
		return ctx
	}

	cf.lock.Lock()
	cf.fields = append(cf.fields, fields)
	cf.lock.Unlock()
	return ctx
}

func Info(ctx context.Context, msg interface{}, fields ...Fields) {
	for _, logger := range loggers {
		logger.Info(ctx, msg, fields...)
	}
}

func Warn(ctx context.Context, msg interface{}, fields ...Fields) {
	for _, logger := range loggers {
		logger.Warn(ctx, msg, fields...)
	}
}

func Error(ctx context.Context, msg interface{}, fields ...Fields) {
	for _, logger := range loggers {
		logger.Error(ctx, msg, fields...)
	}
}

func Fatal(ctx context.Context, msg interface{}, fields ...Fields) {
	for _, logger := range loggers {
		logger.Fatal(ctx, msg, fields...)
	}
}
