package logx

import (
	"fmt"

	"github.com/Sirupsen/logrus"
	"github.com/stvp/rollbar"
)

var (
	logrusToRollbarLevels = map[logrus.Level]string{
		logrus.FatalLevel: rollbar.CRIT,
		logrus.PanicLevel: rollbar.CRIT,
		logrus.ErrorLevel: rollbar.ERR,
		logrus.WarnLevel:  rollbar.WARN,
		logrus.InfoLevel:  rollbar.INFO,
		logrus.DebugLevel: rollbar.DEBUG,
	}
)

func NewRollbarHook(token string, env string) logrus.Hook {
	if token == "" || env == "" {
		return nil
	}

	rollbar.Token = token
	rollbar.Environment = env
	return &rollbarHookImpl{}
}

type rollbarHookImpl struct{}

func (h *rollbarHookImpl) Fire(e *logrus.Entry) error {
	level, ok := logrusToRollbarLevels[e.Level]
	if !ok {
		return fmt.Errorf("Unknown level: %s", e.Level)
	}

	err := fmt.Errorf(e.Message)

	var stack rollbar.Stack
	var fields []*rollbar.Field
	for k, v := range e.Data {
		if k == "stack" {
			if s, ok := v.(rollbar.Stack); ok {
				stack = s
				continue
			}
		}

		fields = append(fields, &rollbar.Field{
			Name: k,
			Data: v,
		})
	}

	if len(stack) > 0 {
		rollbar.ErrorWithStack(level, err, stack, fields...)
	} else {
		rollbar.ErrorWithStackSkip(level, err, 5, fields...)
	}
	return nil
}

func (h *rollbarHookImpl) Levels() []logrus.Level {
	return []logrus.Level{
		logrus.PanicLevel,
		logrus.FatalLevel,
		logrus.ErrorLevel,
	}
}
