package main

import (
	"bytes"
	"crypto/hmac"
	"crypto/rand"
	"crypto/sha1"
	"encoding/base64"
	"fmt"
	"net/http"
	"sort"
	"strconv"
	"strings"
	"time"
)

// TODO: Migrate to sandstorm, rotate these creds.
var (
	oAuthConsumerKey    = "VDc1iNNsJpBTnkvrQFa1"
	oAuthConsumerSecret = "6eTikUeOmGg58GOCFoRX8WSUaztE0XhVPZxD7vQ3"
	oAuthAccessToken    = "XQKogdCQhYg5A4mXdqc2"
	oAuthAccessSecret   = "whDch8ILZcAJn2LfAeEbs09yBSvLyqeXMWxrGiCP"
)

// Largely copied this code from
// https://github.com/dghubble/oauth1/blob/master/auther.go
// We can't use the library in-place because it doesn't export any of the
// resources we need to use to get the Auth header before the request gets sent

const (
	authorizationHeaderParam  = "Authorization"
	authorizationPrefix       = "OAuth " // trailing space is intentional
	oauthConsumerKeyParam     = "oauth_consumer_key"
	oauthNonceParam           = "oauth_nonce"
	oauthSignatureParam       = "oauth_signature"
	oauthSignatureMethodParam = "oauth_signature_method"
	oauthTimestampParam       = "oauth_timestamp"
	oauthTokenParam           = "oauth_token"
	oauthVersionParam         = "oauth_version"
	oauthCallbackParam        = "oauth_callback"
	oauthVerifierParam        = "oauth_verifier"
	defaultOauthVersion       = "1.0"
	contentType               = "Content-Type"
	formContentType           = "application/x-www-form-urlencoded"
)

// GetOAuthHeader returns the Authorization header required to sign a request
//
// Usage:
//
//   req, err := NewRequest("POST", createCasePath, bytes.NewBuffer(body))
//   ... setup stuff for the request
//   header, err := GetOAuthHeader(req)
//   req.Header.Add("Authorization", header)
func GetOAuthHeader(req *http.Request) (string, error) {
	// We do this as a stand-alone method rather than modifying the request in
	// client.Do, since Do isn't supposed to modify the request and chitin already
	// copies the request internally, so I think this is better than causing
	// yet _another_ internal copy.
	if oAuthAccessToken == "" || oAuthAccessSecret == "" ||
		oAuthConsumerKey == "" || oAuthConsumerSecret == "" {
		return "", fmt.Errorf("Retrieving an OAuth header requires all OAuth options to be set.")
	}

	oauthParams := commonOAuthParams()
	oauthParams[oauthTokenParam] = oAuthAccessToken

	params, err := collectParameters(req, oauthParams)
	if err != nil {
		return "", err
	}
	signatureBase := signatureBase(req, params)
	signature, err := sign(signatureBase)
	if err != nil {
		return "", err
	}
	oauthParams[oauthSignatureParam] = signature

	return authHeaderValue(oauthParams), nil
}

func sign(message string) (string, error) {
	signingKey := strings.Join([]string{oAuthConsumerSecret, oAuthAccessSecret}, "&")
	mac := hmac.New(sha1.New, []byte(signingKey))
	mac.Write([]byte(message))
	signatureBytes := mac.Sum(nil)
	return base64.StdEncoding.EncodeToString(signatureBytes), nil
}

func commonOAuthParams() map[string]string {
	return map[string]string{
		oauthConsumerKeyParam:     oAuthConsumerKey,
		oauthSignatureMethodParam: "HMAC-SHA1",
		oauthTimestampParam:       strconv.FormatInt(time.Now().Unix(), 10),
		oauthNonceParam:           nonce(),
		oauthVersionParam:         defaultOauthVersion,
	}
}

func nonce() string {
	b := make([]byte, 32)
	rand.Read(b)
	return base64.StdEncoding.EncodeToString(b)
}

// collectParameters collects request parameters from the request query, OAuth
// parameters (which should exclude oauth_signature), and the request body
// provided the body is single part, form encoded, and the form content type
// header is set. The returned map of collected parameter keys and values
// follow RFC 5849 3.4.1.3, except duplicate parameters are not supporte
func collectParameters(req *http.Request, oauthParams map[string]string) (map[string]string, error) {
	// add oauth, query, and body parameters into params
	params := map[string]string{}
	for key, value := range req.URL.Query() {
		// most backends do not accept duplicate query keys
		params[key] = value[0]
	}
	// TODO: The original source, dghubble/oauth1, supports Body params here,
	// but it also modifies req.Body which we want to avoid, so figure out a way
	// to support body params
	for key, value := range oauthParams {
		params[key] = value
	}
	return params, nil
}

// authHeaderValue formats OAuth parameters according to RFC 5849 3.5.1. OAuth
// params are percent encoded, sorted by key (for testability), and joined by
// "=" into pairs. Pairs are joined with a ", " comma separator into a header
// string.
// The given OAuth params should include the "oauth_signature" key.
func authHeaderValue(oauthParams map[string]string) string {
	pairs := sortParameters(encodeParameters(oauthParams), `%s="%s"`)
	return authorizationPrefix + strings.Join(pairs, ", ")
}

// signatureBase combines the uppercase request method, percent encoded base
// string URI, and normalizes the request parameters int a parameter string.
// Returns the OAuth1 signature base string according to RFC5849 3.4.1.
func signatureBase(req *http.Request, params map[string]string) string {
	method := strings.ToUpper(req.Method)
	baseURL := baseURI(req)
	parameterString := normalizedParameterString(params)
	// signature base string constructed accoding to 3.4.1.1
	baseParts := []string{method, percentEncode(baseURL), percentEncode(parameterString)}
	return strings.Join(baseParts, "&")
}

// parameterString normalizes collected OAuth parameters (which should exclude
// oauth_signature) into a parameter string as defined in RFC 5894 3.4.1.3.2.
// The parameters are encoded, sorted by key, keys and values joined with "&",
// and pairs joined with "=" (e.g. foo=bar&q=gopher).
func normalizedParameterString(params map[string]string) string {
	return strings.Join(sortParameters(encodeParameters(params), "%s=%s"), "&")
}

// baseURI returns the base string URI of a request according to RFC 5849
// 3.4.1.2. The scheme and host are lowercased, the port is dropped if it
// is 80 or 443, and the path minus query parameters is include
func baseURI(req *http.Request) string {
	scheme := strings.ToLower(req.URL.Scheme)
	host := strings.ToLower(req.URL.Host)
	if hostPort := strings.Split(host, ":"); len(hostPort) == 2 && (hostPort[1] == "80" || hostPort[1] == "443") {
		host = hostPort[0]
	}
	// TODO: use req.URL.EscapedPath() once Go 1.5 is more generally adopted
	// For now, hacky workaround accomplishes the same internal escaping mode
	// escape(u.Path, encodePath) for proper compliance with the OAuth1 spec.
	path := req.URL.Path
	if path != "" {
		path = strings.Split(req.URL.RequestURI(), "?")[0]
	}
	return fmt.Sprintf("%v://%v%v", scheme, host, path)
}

// sortParameters sorts parameters by key and returns a slice of key/value
// pairs formatted with the given format string (e.g. "%s=%s").
func sortParameters(params map[string]string, format string) []string {
	// sort by key
	keys := make([]string, len(params))
	i := 0
	for key := range params {
		keys[i] = key
		i++
	}
	sort.Strings(keys)
	// parameter join
	pairs := make([]string, len(params))
	for i, key := range keys {
		pairs[i] = fmt.Sprintf(format, key, params[key])
	}
	return pairs
}

// encodeParameters percent encodes parameter keys and values according to
// RFC5849 3.6 and RFC3986 2.1 and returns a new map.
func encodeParameters(params map[string]string) map[string]string {
	encoded := map[string]string{}
	for key, value := range params {
		encoded[percentEncode(key)] = percentEncode(value)
	}
	return encoded
}

// percentEncode percent encodes a string according to RFC 3986 2.1.
func percentEncode(input string) string {
	var buf bytes.Buffer
	for _, b := range []byte(input) {
		// if in unreserved set
		if shouldEscape(b) {
			buf.Write([]byte(fmt.Sprintf("%%%02X", b)))
		} else {
			// do not escape, write byte as-is
			buf.WriteByte(b)
		}
	}
	return buf.String()
}

// shouldEscape returns false if the byte is an unreserved character that
// should not be escaped and true otherwise, according to RFC 3986 2.1.
func shouldEscape(c byte) bool {
	// RFC3986 2.3 unreserved characters
	if 'A' <= c && c <= 'Z' || 'a' <= c && c <= 'z' || '0' <= c && c <= '9' {
		return false
	}
	switch c {
	case '-', '.', '_', '~':
		return false
	}
	// all other bytes must be escaped
	return true
}
