import sinon from 'sinon';
import { DEFAULT_STATS } from 'state/stats';
import { BackendBlank } from './blank-backend';

const DEFAULT_QUALITY = {
    name: '',
    group: '',
    codecs: '',
    bitrate: 0,
    width: 0,
    height: 0,
    framerate: 0,
};

export class BackendMediaPlayer extends BackendBlank {
    constructor() {
        super();
        this.quality = DEFAULT_QUALITY;
        this.stats = DEFAULT_STATS;
        this.videoInfo = {};
        this.manifestInfo = {};
        this.media = document.createElement('video');
        sinon.spy(this, 'addEventListener');
    }

    getHTMLVideoElement() {
        return this.media;
    }

    load() {}

    _setPlaybackStatistics(stats) {
        this.stats = stats;
    }

    _setVideoInfo(videoInfo) {
        this.videoInfo = videoInfo;
    }

    _setManifestInfo(manifestInfo) {
        this.manifestInfo = manifestInfo;
    }

    _setMedia(media) {
        this.media = media;
    }

    getAutoSwitchQuality() {
        return true;
    }

    setQuality(quality) {
        this.quality = quality;
    }

    getQuality() {
        return this.quality;
    }

    getBufferDuration() {
        return this.stats.bufferSize;
    }

    getBuffered() {
        return {
            start: 0,
            end: 0,
        };
    }

    getManifestInfo() {
        return this.manifestInfo;
    }

    getPlaybackRate() {
        return this.stats.playbackRate;
    }

    getVideoFrameRate() {
        return this.stats.fps;
    }

    getVideoBitRate() {
        return this.stats.bitrate;
    }

    getDuration() {
        return 10;
    }

    getVolume() {
        return this.media.volume;
    }

    getDroppedFrames() {
        return this.stats.skippedFrames;
    }

    getTranscoderLatency() {
        return this.stats.hlsLatencyEncoder;
    }

    getBroadcasterLatency() {
        return this.stats.hlsLatencyBroadcaster;
    }

    getVideoWidth() {
        return this.stats.videoWidth;
    }

    getVideoHeight() {
        return this.stats.videoHeight;
    }

    getDisplayWidth() {
        return this.stats.displayWidth;
    }

    getDisplayHeight() {
        return this.stats.displayHeight;
    }

    getPosition() {
        return 0;
    }

    delete() {}

    getVersion() {
        return 'v1.0.0';
    }

    isMuted() {
        return this.media.muted;
    }

    play() {
        return this.media.play();
    }

    pause() {
        return this.media.pause();
    }

    setVolume(volume) {
        this.media.volume = volume;
    }

    setMuted(mute) {
        this.media.muted = mute;
    }

    getPlayerState() {
        return PlayerState.IDLE;
    }

    seekTo(time) {
        this.media.currentTime = time;
    }

    isSeeking() {
        return false;
    }

    getABSStats() {
        return {};
    }

    getBackend() {
        return BACKEND_MEDIA_PLAYER;
    }
}

export const BACKEND_MEDIA_PLAYER = 'mediaplayer';

export const PlayerEvent = Object.freeze({
    INITIALIZED: 'initialized',
    QUALITY_CHANGED: 'qualityChanged',
    DURATION_CHANGED: 'durationChanged',
    PLAYBACK_RATE_CHANGED: 'playbackRateChanged',
    REBUFFERING: 'rebuffering',
    ERROR: 'mediaplayererror',
    TIME_UPDATE: 'timeupdate',
    BUFFER_UPDATE: 'bufferupdate',
    SEEK_COMPLETED: 'seekCompleted',
    TRACKING: 'tracking',
});

export const MetadataEvent = Object.freeze({
    ID3: 'id3',
    CAPTION: 'caption',
    SPLICE_OUT: 'spliceOut',
    SPLICE_IN: 'spliceIn',
});

export const PlayerState = Object.freeze({
    IDLE: 'Idle',
    READY: 'Ready',
    BUFFERING: 'Buffering',
    PLAYING: 'Playing',
    ENDED: 'Ended',
});

export const ErrorType = Object.freeze({
    GENERIC: 'Error',

    /** Method or feature not supported */
    NOT_SUPPORTED: 'ErrorNotSupported',

    /** No source present for the operation */
    NO_SOURCE: 'ErrorNoSource',

    /** Data or input is invalid for the operation */
    INVALID_DATA: 'ErrorInvalidData',

    /** Class or object is an invalid state */
    INVALID_STATE: 'ErrorInvalidState',

    /** Method parameter is invalid */
    INVALID_PARAMETER: 'ErrorInvalidParameter',

    /** Method or operation timed out */
    TIMEOUT: 'ErrorTimeout',

    /** Output is invalid for the operation */
    INVALID_OUTPUT: 'ErrorInvalidOutput',

    /** An unexpected internal error has occurred. */
    INTERNAL: 'ErrorInternal',

    /** Unspecified Network error */
    NETWORK: 'ErrorNetwork',

    /** Error indicating a network I/O failure */
    NETWORK_IO: 'ErrorNetworkIO',

    /** Error indicating a network resource is not authorized */
    AUTHORIZATION: 'ErrorAuthorization',

    /** Error indicating a network resource is not available */
    NOT_AVAILABLE: 'ErrorNotAvailable',
});

BackendMediaPlayer.canPlay = () => true;

export const ErrorSource = Object.freeze({
    UNKNOWN: 'Unspecified',
    FILE: 'File',
    PLAYLIST: 'Playlist',
    SEGMENT: 'Segment',
    SOURCE: 'Source',
    DECODER: 'Decode',
    RENDERER: 'Render',
});
