import * as AdsActions from 'actions/ads';
import { AdsRequestContext } from 'ads/adscontext';
import { PREROLL } from 'ads/ima-tags';
import { PLAYER_POPOUT } from 'util/player-type';
import { localStore } from 'util/storage';
import assign from 'lodash/assign';
import * as api from 'test-utils/utils/api';
import { createRandomStr } from 'test-utils/utils/create-random-string';
import { LiveTwitchContentStream } from 'stream/twitch-live';
import { VODTwitchContentStream } from 'stream/twitch-vod';
import { init as initStore } from 'state';

const { AdContentTypes, AdRollTypes } = AdsActions;

const DEFAULT_STATE = Object.freeze({
    ads: {
        adblock: false,
    },
    adsManager: {},
    env: {
        playerType: PLAYER_POPOUT,
    },
    experiments: {
        get() {
            return Promise.resolve('');
        },
    },
    manifestInfo: {
        suppress: false,
    },
    stream: {},
    window: {
        document: {
            referrer: '',
        },
        addEventListener: jest.fn(),
        location: {
            href: '',
        },
        top: {
            location: {
                hostname: 'media.curse.com',
            },
        },
        clearTimeout() {},
        setTimeout() {},
    },
});

describe('actions | ads', () => {
    beforeAll(() => {
        api.setLoggedIn(true);
        api.expectUserInfo();
    });

    describe('requestAds - when a user is watching a live stream', () => {
        const channelName = 'monstercat';
        let store;
        let adsManager;

        beforeEach(() => {
            localStore.set('lastAdDisplay', 999999999999);
            adsManager = Object.freeze({
                requestAds: jest.fn(),
                sdk: 'sdkname',
            });
            store = initStore(assign({}, DEFAULT_STATE, {
                adsManager,
                stream: new LiveTwitchContentStream(channelName),
            }));

            const channelInfo = api.expectChannelInfo(channelName);
            api.expectChannelAPIInfo(channelName);
            api.expectChannelViewerInfo(channelName);
            api.expectCommunitiesInfo(channelInfo._id);
            api.expectChannelAdProperties(channelInfo._id);
        });

        test('generates a request context with proper ad properties', function() {
            const action = AdsActions.requestAds(PREROLL, 45);
            const dispatch = jest.fn();

            return action(dispatch, store.getState).then(() => {
                const dispatchMock = dispatch.mock;
                const requestAdsMock = adsManager.requestAds.mock;

                expect(dispatchMock.calls.length).toBe(0);
                expect(requestAdsMock.calls.length).toBe(1);

                const adsRequestContext = requestAdsMock.calls[0][0];
                expect(adsRequestContext instanceof AdsRequestContext).toBe(true);
                expect(adsRequestContext.vodAdsEnabled).toBe(true);
            });
        });

        test('generates a request context and requests ads with it', function() {
            const lastAdDisplay = parseInt(createRandomStr(), 36);
            localStore.set('lastAdDisplay', lastAdDisplay);

            const action = AdsActions.requestAds(PREROLL, 45);
            const dispatch = jest.fn();

            return action(dispatch, store.getState).then(() => {
                const dispatchMock = dispatch.mock;
                const requestAdsMock = adsManager.requestAds.mock;

                expect(dispatchMock.calls.length).toBe(0);
                expect(requestAdsMock.calls.length).toBe(1);

                const adsRequestContext = requestAdsMock.calls[0][0];
                expect(adsRequestContext instanceof AdsRequestContext).toBe(true);

                expect(adsRequestContext.adType).toBe(PREROLL);
                expect(adsRequestContext.duration).toBe(45);
                expect(adsRequestContext.lastAdDisplay).toBe(lastAdDisplay);
            });
        });

        test('if forced is true, generates correct request context and requests ads', function() {
            const forced = true;
            const lastAdDisplay = parseInt(createRandomStr(), 36);
            localStore.set('lastAdDisplay', lastAdDisplay);
            const dispatch = jest.fn();

            const action = AdsActions.requestAds(PREROLL, 45, forced);

            return action(dispatch, store.getState).then(() => {
                const dispatchMock = dispatch.mock;
                const requestAdsMock = adsManager.requestAds.mock;

                expect(dispatchMock.calls.length).toBe(0);
                expect(requestAdsMock.calls.length).toBe(1);

                const adsRequestContext = requestAdsMock.calls[0][0];
                expect(adsRequestContext instanceof AdsRequestContext).toBe(true);

                expect(adsRequestContext.adType).toBe(PREROLL);
                expect(adsRequestContext.duration).toBe(45);
                expect(adsRequestContext.lastAdDisplay).toBe(lastAdDisplay);
                expect(adsRequestContext.forced).toBe(forced);
            });
        });

        test('does not call requestAds on the ads manager when suppress is true', function() {
            const customStore = initStore(assign({}, DEFAULT_STATE, {
                adsManager,
                stream: new LiveTwitchContentStream(channelName),
                manifestInfo: {
                    suppress: true,
                },
            }));
            const dispatch = jest.fn();
            const action = AdsActions.requestAds(PREROLL, 45);

            return action(dispatch, customStore.getState).then(() => {
                const requestAdsMock = adsManager.requestAds.mock;
                expect(requestAdsMock.calls.length).toBe(0);
            });
        });

        test('getRequestContext works with options', function() {
            const options = {
                channel: channelName,
            };
            const action = AdsActions.getRequestContext(
                PREROLL,
                45,
                store.getState(),
                false,
                '',
                options
            );
            return action.then(context => {
                expect(context).toBeTruthy();
                expect(context.channel).toBe(channelName);
            });
        });
    });

    describe('requestAds - when a user is watching a VOD', () => {
        const channelName = 'streamerhouse';
        const videoId = 'v1234567';
        let store;
        let adsManager;

        beforeEach(() => {
            localStore.set('lastAdDisplay', 999999999999);
            adsManager = Object.freeze({
                requestAds: jest.fn(),
                sdk: 'sdkname',
            });
            store = initStore(assign({}, DEFAULT_STATE, {
                adsManager,
                stream: new VODTwitchContentStream(videoId),
                streamMetadata: {
                    broadcast_id: '12345', // eslint-disable-line camelcase
                    name: 'Some VOD Title',
                    type: 'archive',
                },
                playback: {
                    duration: 180,
                },
            }));

            const channelInfo = api.expectChannelInfo(channelName);
            api.expectVideoInfo(videoId, {
                channel: {
                    name: channelName,
                },
            });
            api.expectChannelAPIInfo(channelName);
            api.expectChannelViewerInfo(channelName);
            api.expectCommunitiesInfo(channelInfo._id);
            api.expectChannelAdProperties(channelInfo._id);
        });

        test('if forced is true, generates correct request context and requests ads', function() {
            const forced = true;
            const lastAdDisplay = parseInt(createRandomStr(), 36);
            localStore.set('lastAdDisplay', lastAdDisplay);
            const dispatch = jest.fn();

            const action = AdsActions.requestAds(PREROLL, 45, forced);

            return action(dispatch, store.getState).then(() => {
                const dispatchMock = dispatch.mock;
                const requestAdsMock = adsManager.requestAds.mock;

                expect(dispatchMock.calls.length).toBe(0);
                expect(requestAdsMock.calls.length).toBe(1);

                const adsRequestContext = requestAdsMock.calls[0][0];
                expect(adsRequestContext instanceof AdsRequestContext).toBe(true);

                expect(adsRequestContext.adType).toBe(PREROLL);
                expect(adsRequestContext.duration).toBe(45);
                expect(adsRequestContext.lastAdDisplay).toBe(lastAdDisplay);
                expect(adsRequestContext.forced).toBe(forced);
            });
        });

        test('generates a request context and requests ads with it', function() {
            const lastAdDisplay = parseInt(createRandomStr(), 36);
            localStore.set('lastAdDisplay', lastAdDisplay);
            const dispatch = jest.fn();

            const action = AdsActions.requestAds(PREROLL, 45);

            return action(dispatch, store.getState).then(() => {
                const dispatchMock = dispatch.mock;
                const requestAdsMock = adsManager.requestAds.mock;

                expect(dispatchMock.calls.length).toBe(0);
                expect(requestAdsMock.calls.length).toBe(1);

                const adsRequestContext = requestAdsMock.calls[0][0];
                expect(adsRequestContext instanceof AdsRequestContext).toBe(true);

                expect(adsRequestContext.adType).toBe(PREROLL);
                expect(adsRequestContext.duration).toBe(45);
                expect(adsRequestContext.lastAdDisplay).toBe(lastAdDisplay);
            });
        });

        test('does not call requestAds on the ads manager when suppress is true', function() {
            const customStore = initStore(assign({}, DEFAULT_STATE, {
                adsManager,
                stream: new LiveTwitchContentStream(channelName),
                manifestInfo: {
                    suppress: true,
                },
            }));
            const dispatch = jest.fn();
            const action = AdsActions.requestAds(PREROLL, 45);

            return action(dispatch, customStore.getState).then(() => {
                const requestAdsMock = adsManager.requestAds.mock;
                expect(requestAdsMock.calls.length).toBe(0);
            });
        });

        test('getRequestContext works with options', function() {
            const options = {
                video: videoId,
            };
            const action = AdsActions.getRequestContext(
                PREROLL,
                45,
                store.getState(),
                false,
                '',
                options
            );
            return action.then(context => {
                expect(context).toBeTruthy();
                expect(context.vod.id).toBe(videoId);
            });
        });
    });

    describe('setAdblockDetected', () => {
        test('creates a correctly formatted Set Adblock Detected action', function() {
            const action = AdsActions.setAdblockDetected(true);

            expect(action).toEqual({
                type: AdsActions.ACTION_SET_ADBLOCK_DETECTED,
                detected: true,
            });
        });
    });

    describe('setAdClickThrough', () => {
        test('creates a correctly formatted Set Ad clickthrough action', function() {
            const action = AdsActions.setAdClickThrough('test');

            expect(action).toEqual({
                type: AdsActions.ACTION_SET_CLICKTHROUGH_URL,
                URL: 'test',
            });
        });
    });

    describe('setVodMidrollBlacklisted', () => {
        test('creates a orrectly formatted Set Vodmidroll Blacklist action', function() {
            const action = AdsActions.setVodMidrollArchiveSetting('test');

            expect(action).toEqual({
                type: AdsActions.ACTION_SET_VOD_MIDROLL_ARCHIVE_SETTING,
                setting: 'test',
            });
        });
    });

    describe('setCurrentAdMetadata', () => {
        // eslint-disable-next-line max-len
        test('creates a correctly formatted Set Current Ad Metadata action for a properly formatted ad parameter', function() {
            const adMetadata = {
                contentType: AdContentTypes.IMA,
                rollType: AdRollTypes.PREROLL,
            };
            const action = AdsActions.setCurrentAdMetadata(adMetadata);

            expect(action).toEqual({
                type: AdsActions.ACTION_SET_AD_METADATA,
                currentMetadata: adMetadata,
            });
        });

        test('creates an invalid ad action for an improperly formatted ad parameter', function() {
            const INCORRECT_ADS_METADATA = [
                {
                    contentType: AdContentTypes.IMA,
                    rollType: AdRollTypes.NONE,
                },
                {
                    contentType: AdContentTypes.NONE,
                    rollType: AdRollTypes.PREROLL,
                },
                {
                    contentType: AdContentTypes.NOTACONTENTTYPE,
                    rollType: AdRollTypes.NOTAROLLTYPE,
                },
                {
                    contentType: AdContentTypes.IMA,
                },
                {
                    rollType: AdRollTypes.PREROLL,
                },
            ];

            INCORRECT_ADS_METADATA.forEach(adMetadata => {
                const action = AdsActions.setCurrentAdMetadata(adMetadata);

                expect(action).toEqual(AdsActions.INVALID_AD_ACTION);
            });
        });
    });

    describe('clearcurrentMetadataAd', () => {
        test('creates a correctly formatted Set Current Ad Metadata Action', function() {
            const action = AdsActions.clearCurrentAdMetadata();

            expect(action).toEqual({
                type: AdsActions.ACTION_SET_AD_METADATA,
                currentMetadata: {
                    contentType: AdContentTypes.NONE,
                    rollType: AdRollTypes.NONE,
                },
            });
        });
    });

    describe('imaScriptLoaded', () => {
        test('sets the status to true if script loaded', function() {
            const action = AdsActions.imaScriptLoaded(true);

            const expectedAction = {
                type: AdsActions.ACTION_IMA_SCRIPT_LOADED,
                imaScriptLoaded: true,
            };

            expect(action).toEqual(expectedAction);
        });

        test('sets the status to false if script failed', function() {
            const action = AdsActions.imaScriptLoaded(false);

            const expectedAction = {
                type: AdsActions.ACTION_IMA_SCRIPT_LOADED,
                imaScriptLoaded: false,
            };

            expect(action).toEqual(expectedAction);
        });
    });
});
