import * as LanguageActions from 'actions/lang';
import * as api from 'test-utils/utils/api';
import { DEFAULT_LANGUAGE } from 'lang/i18n';
import isFunction from 'lodash/isFunction';
import { cookie } from 'cookie_js';

describe('actions | lang', function() {
    describe('setLanguage', function() {
        test('creates an appropriate action object', function() {
            const lang = {};
            const action = LanguageActions.setLanguage(lang);

            expect(action.type).toBe(LanguageActions.ACTION_SET_LANGUAGE);
            expect(action.lang).toBe(lang);
        });
    });

    describe('loadLanguage', function() {
        let language;
        const getState = () => {
            return {
                lang: language,
            };
        };

        beforeAll(() => {
            language = DEFAULT_LANGUAGE;
        });

        test('returns a function', function() {
            const action = LanguageActions.loadLanguage('de');
            expect(isFunction(action)).toBeTruthy();
        });

        test('eventually sets the language if it is changed', function() {
            api.mock('/localization/de-DE/strings.json', { body: {} });
            const action = LanguageActions.loadLanguage('de');
            const dispatch = jest.fn();

            const result = action(dispatch, getState);

            return result.then(() => {
                expect(dispatch.mock.calls[0][0].type).toBe(LanguageActions.ACTION_SET_LANGUAGE);
                expect(dispatch.mock.calls[0][0].lang.shortCode).toBe('de');
                expect(dispatch.mock.calls[0][0].lang.langCode).toBe('de-DE');
            });
        });

        test('does nothing if the language is unchanged', function() {
            const action = LanguageActions.loadLanguage('en');
            const dispatch = jest.fn();

            const result = action(dispatch, getState);

            return result.then(() => {
                expect(dispatch.mock.calls.length).toBe(0);
            });
        });
    });

    describe('loadDefaultLang', function() {
        let language;
        let navigator;
        let loadsLanguage;

        const getState = () => {
            return {
                lang: language,
            };
        };

        beforeAll(() => {
            api.clearFakeResponses();
            language = DEFAULT_LANGUAGE;
            navigator = {};

            loadsLanguage = (action, lang) => {
                api.mock(`/localization/${lang}/strings.json`, { body: {} });
                const dispatch = jest.fn();
                const result = action(dispatch, getState);

                return result.then(() => {
                    if (dispatch.mock.calls[0][0].lang.langCode !== lang) {
                        return Promise.reject();
                    }
                });
            };
        });

        test('should first check the cookie', function() {
            cookie.set('language', 'de');

            const action = LanguageActions.loadDefaultLang(navigator, null);
            return loadsLanguage(action, 'de-DE');
        });

        test('should check the first language on the `navigator` object', function() {
            navigator.languages = ['de-DE', 'es'];

            const action = LanguageActions.loadDefaultLang(navigator, null);
            return loadsLanguage(action, 'de-DE');
        });

        test('should check the language defined on the `navigator` object', function() {
            navigator.language = 'de-DE';

            const action = LanguageActions.loadDefaultLang(navigator, null);
            return loadsLanguage(action, 'de-DE');
        });

        test('should check the `userLanguage` property on the `navigator` object', function() {
            navigator.userLanguage = 'de-ch';

            const action = LanguageActions.loadDefaultLang(navigator, null);
            return loadsLanguage(action, 'de-DE');
        });

        test('should use the provided default language otherwise', function() {
            const action = LanguageActions.loadDefaultLang(navigator, 'de');
            return loadsLanguage(action, 'de-DE');
        });
    });
});
