import isFunction from 'lodash/isFunction';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { localStore } from 'util/storage';
import * as PlaybackActions from 'actions/playback';
import { createRandomStr } from 'test-utils/utils/create-random-string';
import * as QualityActions from 'actions/quality';

const SOURCE_QUALITY = 'chunked';
const HIGH_QUALITY = 'high';
const MEDIUM_QUALITY = 'medium';
const MEDIUM_BITRATE = 992000;
const HIGH_BITRATE = 1760000;

describe('actions/quality', function() {
    let state;

    afterEach(() => {
        localStore.clear();
    });

    describe('initializeQuality', function() {
        beforeEach(function() {
            state = {
                quality: {
                    selected: 'high',
                    bitrate: HIGH_BITRATE,
                },
            };
        });

        test('selects the quality based on localStore, if one exists', function() {
            localStore.set('quality', 'medium');
            localStore.set('quality-bitrate', MEDIUM_BITRATE);
            const action = QualityActions.initializeQuality();
            const dispatch = sinon.spy();

            action(dispatch, () => state);

            expect(dispatch.callCount).toBe(1);
            expect(dispatch.calledWith({
                type: QualityActions.ACTION_SELECT_QUALITY,
                quality: 'medium',
                bitrate: MEDIUM_BITRATE,
            })).toBeTruthy();
        });

        test('defaults to the pre-configured default quality', function() {
            expect(!localStore.has('quality')).toBeTruthy();

            const action = QualityActions.initializeQuality();
            const dispatch = sinon.spy();

            action(dispatch, () => state);

            expect(dispatch.callCount).toBe(1);
            expect(dispatch.calledWith({
                type: QualityActions.ACTION_SELECT_QUALITY,
                quality: state.quality.selected,
                bitrate: state.quality.bitrate,
            })).toBeTruthy();
        });
    });

    test('setABSAvailability is formatted correctly', function() {
        const action = QualityActions.setABSAvailability(true);

        const expectedAction = {
            type: QualityActions.ACTION_SET_ABS_AVAILABILITY,
            absAvailable: true,
        };

        expect(action).toEqual(expectedAction);
    });

    describe('selectQuality', function() {
        let fakeAnalytics;
        beforeEach(function() {
            fakeAnalytics = {
                trackEvent: sinon.spy(),
            };
            state = {
                analyticsTracker: fakeAnalytics,
                manifestInfo: {
                    // eslint-disable-next-line camelcase
                    serving_id: 'abc123',
                },
                quality: {
                    selected: MEDIUM_QUALITY,
                    current: MEDIUM_QUALITY,
                    bitrate: MEDIUM_BITRATE,
                },
                stream: {
                    restrictedBitrates: [],
                },
            };
        });

        test('dispatches a Select Quality action', function() {
            const dispatch = sinon.spy();
            const action = QualityActions.selectQuality(HIGH_QUALITY, HIGH_BITRATE);

            expect(isFunction(action)).toBeTruthy();
            action(dispatch, () => state);

            expect(dispatch.callCount).toBe(1);
            expect(dispatch.firstCall.calledWith({
                type: QualityActions.ACTION_SELECT_QUALITY,
                quality: HIGH_QUALITY,
                bitrate: HIGH_BITRATE,
            })).toBeTruthy();
        });

        test('send a tracking event for quality change request', function() {
            const action = QualityActions.selectQuality(HIGH_QUALITY, HIGH_BITRATE);
            expect(isFunction(action)).toBeTruthy();
            action(function() {}, () => state);

            expect(fakeAnalytics.trackEvent.callCount).toBe(1);
            expect(fakeAnalytics.trackEvent.firstCall.args[0]).toBe('quality_change_request');
            /* eslint-disable camelcase */
            expect(fakeAnalytics.trackEvent.firstCall.args[1]).toEqual({
                prev_quality: state.quality.current,
                new_quality: HIGH_QUALITY,
                serving_id: state.manifestInfo.serving_id,
            });
            /* eslint-enable camelcase */
        });

        test('dispatches error for restricted qualities', function() {
            const dispatch = sinon.spy();
            const state = {
                quality: {
                    selected: SOURCE_QUALITY,
                },
                stream: {
                    restrictedBitrates: [SOURCE_QUALITY],
                },
            };
            const action = QualityActions.selectQuality(SOURCE_QUALITY);

            expect(isFunction(action)).toBeTruthy();
            action(dispatch, () => state);

            expect(dispatch.callCount).toBe(1);
            expect(dispatch.calledWith({
                type: PlaybackActions.ACTION_QUALITY_RESTRICTED_ERROR,
            })).toBeTruthy();
        });
    });

    describe('setPreferredQuality', function() {
        test('returns a thunk action', function() {
            const quality = {
                name: `Quality Name ${createRandomStr()}`,
                group: `qualname_${createRandomStr()}`,
                bandwidth: Math.floor(1e7 * Math.random()),
            };
            const action = QualityActions.setPreferredQuality(quality);

            expect(isFunction(action)).toBeTruthy();
        });

        test('attempts to set the given quality\'s group as the persisted value', function() {
            const quality = {
                name: `Quality Name ${createRandomStr()}`,
                group: `qualname_${createRandomStr()}`,
                bandwidth: Math.floor(1e7 * Math.random()),
            };
            const action = QualityActions.setPreferredQuality(quality);

            expect(localStore.get('quality')).not.toBe(quality.group);
            expect(localStore.get('quality-bitrate')).not.toBe(quality.bandwidth);

            action();

            expect(localStore.get('quality')).toBe(quality.group);
            expect(localStore.get('quality-bitrate')).toBe(quality.bandwidth);
        });
    });

    describe('setCurrentQuality', function() {
        test('setCurrentQuality action is formatted correctly', function() {
            const newQuality = HIGH_QUALITY;
            const expectedResult = {
                type: QualityActions.ACTION_SET_CURRENT_QUALITY,
                quality: newQuality,
                isAuto: false,
            };
            const actualResult = QualityActions.setCurrentQuality(newQuality);

            expect(actualResult).toEqual(expectedResult);
        });
    });

    describe('setQualities', function() {
        beforeEach(function() {
            state = {
                quality: {
                    selected: HIGH_QUALITY,
                    current: HIGH_QUALITY,
                    bitrate: HIGH_BITRATE,
                },
                stream: {
                    restrictedBitrates: [],
                },
            };
        });

        function testMatchingTranscodeQuality(selectedQuality, qualities) {
            test('_getMatchingTranscodeQuality finds the matching transcode quality', function() {
                const matchedQuality = QualityActions._getMatchingTranscodeQuality(selectedQuality, qualities);
                const [expectedQuality] = qualities;
                expect(matchedQuality).toEqual(expectedQuality);
            });
        }

        testMatchingTranscodeQuality('high', [{ group: 'high2' }]);
        testMatchingTranscodeQuality('high2', [{ group: 'high' }]);

        test('_getMatchingTranscodeQuality should return undefined if no match found', function() {
            const noQualityFound = QualityActions._getMatchingTranscodeQuality('garbage', [{ group: 'high2' }]);
            expect(noQualityFound).toBe(undefined);
        });

        test('dispatches a setQualities action', function() {
            const qualities = [
                {
                    group: SOURCE_QUALITY,
                    name: 'Source',
                    bandwidth: 176000,
                    resolution: '1920x1080',
                },
                {
                    group: HIGH_QUALITY,
                    name: 'High',
                    bandwidth: 171600,
                    resolution: '1280x720',
                },
            ];

            const dispatch = sinon.spy();
            const action = QualityActions.setQualities(qualities);

            expect(isFunction(action)).toBeTruthy();
            action(dispatch, () => state);

            expect(dispatch.callCount).toBe(1);
            expect(dispatch.firstCall.calledWith({
                type: QualityActions.ACTION_SET_QUALITIES,
                qualities,
            })).toBeTruthy();
        });

        test('if current quality is unavailable, also dispatches a selectQuality action', function() {
            const qualities = [
                {
                    group: SOURCE_QUALITY,
                    name: 'Source',
                    bandwidth: 176000,
                    resolution: '1920x1080',
                },
            ];

            const dispatch = sinon.spy();
            const action = QualityActions.setQualities(qualities);

            expect(isFunction(action)).toBeTruthy();
            action(dispatch, () => state);

            expect(dispatch.callCount).toBe(2);
            expect(dispatch.firstCall.calledWith({
                type: QualityActions.ACTION_SELECT_QUALITY,
                quality: SOURCE_QUALITY,
                bitrate: 176000,
            })).toBeTruthy();
            expect(dispatch.secondCall.calledWith({
                type: QualityActions.ACTION_SET_QUALITIES,
                qualities,
            })).toBeTruthy();
        });

        describe('setQualities when the default is not available', function() {
            beforeEach(function() {
                state = {
                    quality: {
                        selected: MEDIUM_QUALITY,
                        current: MEDIUM_QUALITY,
                        bitrate: MEDIUM_BITRATE,
                    },
                    stream: {
                        restrictedBitrates: [],
                    },
                };
            });

            // eslint-disable-next-line max-len
            test('if qualities are from a different transcode stack, find and set the matching quality', function() {
                const qualities = [
                    {
                        group: SOURCE_QUALITY,
                        name: 'Source',
                        bandwidth: 3161368,
                        resolution: '1280x720',
                    },
                    {
                        group: 'high2',
                        name: 'High',
                        bandwidth: 2048960,
                        resolution: '1280x720',
                    },
                    {
                        group: 'medium2',
                        name: '480p30',
                        bandwidth: 1375632,
                        resolution: '852×480',
                    },
                ];
                const dispatch = sinon.spy();
                const newState = assign({}, state, {
                    quality: {
                        selected: HIGH_QUALITY,
                        current: HIGH_QUALITY,
                        bitrate: MEDIUM_BITRATE,
                    },
                });
                const action = QualityActions.setQualities(qualities);
                action(dispatch, () => newState);

                expect(dispatch.callCount).toBe(2, 'dispatches two actions');

                expect(dispatch.firstCall.args[0]).toEqual({
                    type: QualityActions.ACTION_SELECT_QUALITY,
                    quality: 'high2',
                    bitrate: 2048960,
                }, 'Selects the matching transcode from the list of qualities');

                expect(dispatch.secondCall.args[0]).toEqual({
                    type: QualityActions.ACTION_SET_QUALITIES,
                    qualities,
                });
            });

            // eslint-disable-next-line max-len
            test('if the exact quality is not present in stream, should choose the max quality less than or equal to selected quality ', function() {
                const qualities = [
                    {
                        group: '720p60',
                        name: '720p60',
                        bandwidth: 3161368,
                        resolution: '1280x720',
                    },
                    {
                        group: '720p30',
                        name: '720p30',
                        bandwidth: 2048960,
                        resolution: '1280x720',
                    },
                    {
                        group: '480p30',
                        name: '480p30',
                        bandwidth: 1375632,
                        resolution: '852×480',
                    },
                    {
                        group: '360p30',
                        name: '360p30',
                        bandwidth: 749072,
                        resolution: '640×360',
                    },
                    {
                        group: '144p30',
                        name: '144p30',
                        bandwidth: 366184,
                        resolution: '256x144',
                    },
                ];

                const dispatch = sinon.spy();
                const action = QualityActions.setQualities(qualities);
                action(dispatch, () => state);

                expect(dispatch.callCount).toBe(
                    2,
                    'Two dispatches expected. One for selecting the default quality and one for setting all qualities'
                );
                expect(dispatch.firstCall.args[0]).toEqual({
                    type: QualityActions.ACTION_SELECT_QUALITY,
                    quality: '360p30',
                    bitrate: 749072,
                }, 'Dispatching action ACTION_SELECT_QUALITY with quality of bandwidth less than MEDIUM_BITRATE');
                expect(dispatch.secondCall.args[0]).toEqual({
                    type: QualityActions.ACTION_SET_QUALITIES,
                    qualities,
                }, 'Dispatching action ACTION_SET_QUALITIES');
            });

            // eslint-disable-next-line max-len
            test('if stream has only one quality but not exact, choose the only quality available', function() {
                const qualities = [
                    {
                        group: '720p60',
                        name: '720p60',
                        bandwidth: 3161368,
                        resolution: '1280x720',
                    },
                ];

                const dispatch = sinon.spy();
                const action = QualityActions.setQualities(qualities);
                action(dispatch, () => state);

                expect(dispatch.callCount).toBe(
                    2,
                    'Two dispatches expected. One for selecting the default quality and one for setting all qualities'
                );
                expect(dispatch.firstCall.args[0]).toEqual({
                    type: QualityActions.ACTION_SELECT_QUALITY,
                    quality: '720p60',
                    bitrate: 3161368,
                }, 'Dispatching action ACTION_SELECT_QUALITY with only quality which is available');
                expect(dispatch.secondCall.args[0]).toEqual({
                    type: QualityActions.ACTION_SET_QUALITIES,
                    qualities,
                }, 'Dispatching action ACTION_SET_QUALITIES');
            });
        });
    });

    test('setClipQualities dispatches a setQualities action with normalized qualities', function() {
        const clipQualities = [
            {
                quality: '1080',
                source: '1080url.mp4',
                // eslint-disable-next-line camelcase
                frame_rate: '60',
            },
            {
                quality: '720',
                source: '720url.mp4',
                // eslint-disable-next-line camelcase
                frame_rate: '60',
            },
            {
                quality: '480',
                source: '480url.mp4',
                // eslint-disable-next-line camelcase
                frame_rate: '30',
            },
            {
                quality: '360',
                source: '360url.mp4',
                // eslint-disable-next-line camelcase
                frame_rate: '30',
            },
        ];

        const expectedQualities = clipQualities.map(clipQuality => {
            return {
                name: `${clipQuality.quality}p`,
                group: `${clipQuality.quality}p${clipQuality.frame_rate}`,
                bitrate: '',
                resolution: QualityActions.CLIP_RESOLUTION_MAP[clipQuality.quality],
                source: clipQuality.source,
            };
        });

        const action = QualityActions.setClipQualities(clipQualities);

        expect(action).toEqual({
            type: QualityActions.ACTION_SET_QUALITIES,
            qualities: expectedQualities,
        });
    });
});
