import find from 'lodash/find';
import times from 'lodash/times';
import difference from 'lodash/difference';
import { stringify } from 'query-string';
import { channelInfo, videoInfo, streamInfo,
         krakenUserInfo, krakenRequest, oauthToken,
         setOAuthToken, setFollowChannel, getFollowChannel,
         setFollowNotifications, krakenRequestv5, overlayExtensionsForChannel,
         getSubscriptionInfo, getFeaturedCollection, postExtensionReport,
         createClip, getChannelAdProperties } from 'api';
import { TEST_COLLECTION_METADATA } from 'test-utils/fixtures/collection';
import * as PlayerType from 'util/player-type';
import { LiveTwitchContentStream } from 'stream/twitch-live';
import * as api from 'test-utils/utils/api';
import { createRandomStr } from 'test-utils/utils/create-random-string';

const API_KRAKEN_VERSION = 'application/vnd.twitchtv.v3+json';
const API_KRAKEN_VERSION_V5 = 'application/vnd.twitchtv.v5+json';
const API_CLIENT_ID = 'jzkbprff40iqj646a697cyrvl0zt2m6';

describe('api', function() {
    beforeEach(function() {
        api.setLoggedIn(false);
    });

    afterEach(function() {
        api.clearFakeResponses();
        // oauthToken is memoized and should be cleared after each test
        oauthToken.cache.clear();
    });

    test('getFeaturedCollection returns recent featured collection id', function() {
        expect.assertions(1);
        const channelId = 12345;
        api.expectFeaturedCollection(channelId, true);

        return getFeaturedCollection(channelId).then(data => {
            expect(data.collections).toEqual([TEST_COLLECTION_METADATA]);
        });
    });

    test('getFeaturedCollection returns empty array if no collections are found', function() {
        expect.assertions(1);
        const channelId = 12345;
        api.expectFeaturedCollection(channelId, false);

        return getFeaturedCollection(channelId).then(data => {
            expect(data.collections).toEqual([]);
        });
    });

    test('getFollowChannel request contains correct headers', function() {
        expect.assertions(2);
        const userId = 1234;
        const channelId = 5678;
        api.expectFollowInfo({
            userId,
            channelId,
            following: true,
            notificationsEnabled: false,
        });

        return getFollowChannel(userId, channelId).then(() => {
            const requestArgs = api.lastCall()[1];
            expect(requestArgs.headers['Client-ID']).toBe(API_CLIENT_ID);
            expect(requestArgs.headers.Accept).toBe(API_KRAKEN_VERSION_V5);
        });
    });

    function testSetFollow(follow) {
        const method = follow ? 'PUT' : 'DELETE';
        test(`setFollowChannel makes a ${method} request when passed ${follow}`, function() {
            expect.assertions(3);
            const userId = 1234;
            const channelId = 5678;
            api.expectSetFollow(follow, {
                channelId,
            });

            return setFollowChannel(userId, channelId, follow).then(() => {
                const requestArgs = api.lastCall()[1];
                expect(requestArgs.headers['Client-ID']).toBe(API_CLIENT_ID);
                expect(requestArgs.headers.Accept).toBe(API_KRAKEN_VERSION_V5);
                expect(requestArgs.method).toBe(method);
            });
        });
    }

    testSetFollow(true);
    testSetFollow(false);

    function testSetNotifications(enableNotifications) {
        // eslint-disable-next-line max-len
        const msg = `setFollowNotifications sends expected form data with PUT request - notifications: ${enableNotifications}`;
        test(msg, function() {
            expect.assertions(4);
            const userId = 1234;
            const channelId = 5678;
            api.expectSetNotifications(enableNotifications, {
                channelId,
            });

            return setFollowNotifications(userId, channelId, enableNotifications).then(() => {
                const requestArgs = api.lastCall()[1];
                expect(requestArgs.headers['Client-ID']).toBe(API_CLIENT_ID);
                expect(requestArgs.headers.Accept).toBe(API_KRAKEN_VERSION_V5);
                expect(requestArgs.method).toBe('PUT');
                expect(requestArgs.body).toBe(`notifications=${enableNotifications}`);
            });
        });
    }

    testSetNotifications(true);
    testSetNotifications(false);

    test('streamInfo request contains correct headers', function() {
        api.expectStreamInfo('monstercat');
        expect.assertions(2);

        return streamInfo('monstercat').then(() => {
            const requestArgs = api.lastCall()[1];

            expect(requestArgs.headers['Client-ID']).toBe(API_CLIENT_ID);
            expect(requestArgs.headers.Accept).toBe(API_KRAKEN_VERSION);
        });
    });

    test('channelInfo request contains correct headers', function() {
        api.expectChannelInfo('monstercat');

        return channelInfo('monstercat').then(() => {
            const requestArgs = api.lastCall()[1];

            expect(requestArgs.headers['Client-ID']).toBe(API_CLIENT_ID);
            expect(requestArgs.headers.Accept).toBe(API_KRAKEN_VERSION);
        });
    });

    test('videoInfo request contains correct headers', function() {
        expect.assertions(2);
        const videoData = api.expectVideoInfo('v123456789', {
            channel: {
                name: 'channelName',
            },
        });
        api.expectChannelInfo(videoData.channel.name);
        return videoInfo('v123456789').then(() => {
            const requestArgs = api.lastCall()[1];

            expect(requestArgs.headers['Client-ID']).toBe(API_CLIENT_ID);
            expect(requestArgs.headers.Accept).toBe(API_KRAKEN_VERSION);
        });
    });

    test('getChannelAdProperties request contains correct headers', function() {
        expect.assertions(2);
        api.setLoggedIn(true);
        const stream = new LiveTwitchContentStream(
            `channel_${createRandomStr().toLowerCase()}`,
            Promise.reject(),
            {}
        );
        const channelInfo = api.expectChannelInfo(stream.channel);
        api.expectChannelAdProperties(channelInfo._id);
        return getChannelAdProperties(stream.channel).then(() => {
            const requestArgs = api.lastCall()[1];

            expect(requestArgs.headers['Client-ID']).toBe(API_CLIENT_ID);
            expect(requestArgs.headers.Accept).toBe(API_KRAKEN_VERSION_V5);
        });
    });

    test('when logged in, should cache the oauthToken when possible', () => {
        expect.assertions(2);
        api.clearFakeResponses();
        api.setLoggedIn(true);
        const promises = times(5, () => oauthToken());

        return Promise.all(promises).
            then(() => {
                expect(api.calls().matched.length).toBe(1);
                expect(api.calls().unmatched.length).toBe(0);
            });
    });

    test('when logged out, should cache the oauthToken response', () => {
        expect.assertions(6);
        const loggedInError = () => expect('should not be logged in').toBe(null);
        const retryToken = () => {
            expect(api.calls().matched.length).toBe(1);
            expect(api.calls().unmatched.length).toBe(0);
            return oauthToken();
        };

        return oauthToken().
            then(loggedInError, retryToken).
            then(loggedInError, retryToken).
            then(loggedInError, retryToken).
            catch(() => {});
    });

    test('getSubscriptionInfo fetches subscriptions data', function() {
        expect.assertions(2);
        api.expectGetSubscriptionInfo({
            userId: 123,
            channelId: 321,
            isSubscribed: true,
        });

        // What the server returns doesn't matter as long as it's a 2** response.
        return getSubscriptionInfo(123, 321).then(() => {
            const requestArgs = api.lastCall()[1];

            expect(requestArgs.headers['Client-ID']).toBe(API_CLIENT_ID);
            expect(requestArgs.headers.Accept).toBe(API_KRAKEN_VERSION_V5);
        });
    });

    test('should not be able to set the OAuth token when it is falsey', function() {
        expect.assertions(2);
        const token = 'currentToken';

        api.clearFakeResponses();
        api.setLoggedIn(true, { token });

        return oauthToken().
            then(response => {
                expect(response.token).toBe(token);

                [null, undefined, ''].forEach(value => {
                    setOAuthToken({
                        player: PlayerType.PLAYER_CURSE,

                        // eslint-disable-next-line camelcase
                        oauth_token: value,
                    });
                });
                return oauthToken();
            }).
            then(response => {
                expect(response.token).toBe(token);
            });
    });

    test('oauthToken request should be made with credentials field', function() {
        expect.assertions(2);
        const token = 'currentToken';

        api.clearFakeResponses();
        api.setLoggedIn(true, { token });
        return oauthToken().
            then(() => {
                const lastRequest = api.lastCall();
                expect(lastRequest[0]).toBe('https://api.twitch.tv/api/viewer/token.json');
                expect(lastRequest[1].credentials).toBe('include');
            });
    });

    test('krakenUserInfo request contains correct headers', function() {
        expect.assertions(2);
        api.expectKrakenUserInfo();

        return krakenUserInfo().then(() => {
            const requestArgs = api.lastCall()[1];

            expect(requestArgs.headers['Client-ID']).toBe(API_CLIENT_ID);
            expect(requestArgs.headers.Accept).toBe(API_KRAKEN_VERSION);
        });
    });

    test('krakenRequest makes requests with correct headers', function() {
        expect.assertions(2);
        api.mock(/kraken\/channels\/monstercat$/, {
            status: 200,
            headers: {},
            body: {},
        });

        return krakenRequest('channels/monstercat').then(() => {
            const requestArgs = api.lastCall()[1];

            expect(requestArgs.headers['Client-ID']).toBe(API_CLIENT_ID);
            expect(requestArgs.headers.Accept).toBe(API_KRAKEN_VERSION);
        });
    });

    function testKrakenRequestv5(options, assertions) {
        expect.assertions(2);
        // eslint-disable-next-line max-len
        test(`krakenRequestv5 makes requests with correct method: ${options.method}`, function() {
            api.mock(/kraken\/channels\/monstercat$/, {
                status: 200,
                headers: {},
                body: {},
            }, {
                method: options.method,
            });

            return krakenRequestv5('channels/monstercat', options).then(() => {
                const requestArgs = api.lastCall()[1];
                expect(requestArgs.headers['Client-ID']).toBe(API_CLIENT_ID);
                expect(requestArgs.headers.Accept).toBe(API_KRAKEN_VERSION_V5);

                if (assertions) {
                    assertions(requestArgs);
                }
            });
        });
    }

    testKrakenRequestv5({
        method: 'GET',
    });

    testKrakenRequestv5({
        method: 'DELETE',
    });

    testKrakenRequestv5({
        method: 'PUT',
        data: { notifications: true },
    }, function(requestArgs) {
        expect(requestArgs.body).toBe('notifications=true');
    });

    describe('setOAuthToken', function() {
        expect.assertions(1);
        const ALL_VALUES = PlayerType.ALL_PLAYER_TYPES;
        const SETTABLE_TYPES = [
            PlayerType.PLAYER_CURSE,
            PlayerType.PLAYER_FRONTPAGE,
            PlayerType.PLAYER_SITE,
            PlayerType.PLAYER_FEED,
        ];
        const UNSETTABLE_TYPES = difference(ALL_VALUES, SETTABLE_TYPES);

        SETTABLE_TYPES.forEach(playerType => {
            test(`should be able to set the OAuth token for type '${playerType}'`, function() {
                const token = createRandomStr();
                setOAuthToken({
                    player: playerType,
                    // eslint-disable-next-line camelcase
                    oauth_token: token,
                });

                return oauthToken().then(actual => {
                    expect(actual).toEqual({ token });
                });
            });
        });

        UNSETTABLE_TYPES.forEach(playerType => {
            expect.assertions(1);
            const DEFAULT_TOKEN = Object.freeze({ token: 'default-token' });

            test(`should not be able to set the OAuth token for type '${playerType}'`, function() {
                api.clearFakeResponses();
                api.setLoggedIn(true, DEFAULT_TOKEN);
                setOAuthToken({
                    player: playerType,
                    // eslint-disable-next-line camelcase
                    oauth_token: createRandomStr(),
                });

                return oauthToken().
                    then(actual => {
                        expect(actual).toEqual(DEFAULT_TOKEN);
                    });
            });
        });
    });

    describe('overlayExtensionsForChannel', function() {
        test('should convert the response into the local representation', function() {
            expect.assertions(1);
            const channelName = `channel_${createRandomStr()}`;
            const channelInfo = api.expectChannelInfo(channelName);
            const extResponse = api.expectOverlayExtensionsForChannel(channelInfo._id, 1);

            return overlayExtensionsForChannel(channelName).then(extConfig => {
                expect(extConfig).toEqual([
                    {
                        id: extResponse.installed_extensions[0].extension.id,
                        name: extResponse.installed_extensions[0].extension.name,
                        sku: extResponse.installed_extensions[0].extension.sku,
                        summary: extResponse.installed_extensions[0].extension.summary,
                        anchor: extResponse.installed_extensions[0].extension.anchor,
                        version: extResponse.installed_extensions[0].extension.version,
                        state: extResponse.installed_extensions[0].extension.state,
                        vendorCode: extResponse.installed_extensions[0].extension.vendor_code,
                        viewerUrl: extResponse.installed_extensions[0].extension.viewer_url,
                        supportsIdentityLinking: extResponse.installed_extensions[0].extension.request_identity_link,
                        lastUserIdentityLinkState: false,
                        token: {
                            token: extResponse.tokens[0].token,
                            permissionsState: 'none',
                            role: 'viewer',
                        },
                    },
                ]);
            });
        });

        test('should skip extensions not activated as overlays', function() {
            expect.assertions(1);
            const channelName = `channel_${createRandomStr()}`;
            const channelInfo = api.expectChannelInfo(channelName);
            const extResponse = api.expectOverlayExtensionsForChannel(channelInfo._id, 2);
            // eslint-disable-next-line camelcase
            extResponse.installed_extensions[0].installation_status.activation_config = {};

            return overlayExtensionsForChannel(channelName).then(extConfig => {
                expect(extConfig).toEqual([
                    {
                        id: extResponse.installed_extensions[1].extension.id,
                        name: extResponse.installed_extensions[1].extension.name,
                        sku: extResponse.installed_extensions[1].extension.sku,
                        summary: extResponse.installed_extensions[1].extension.summary,
                        anchor: extResponse.installed_extensions[1].extension.anchor,
                        version: extResponse.installed_extensions[1].extension.version,
                        state: extResponse.installed_extensions[1].extension.state,
                        vendorCode: extResponse.installed_extensions[1].extension.vendor_code,
                        viewerUrl: extResponse.installed_extensions[1].extension.viewer_url,
                        supportsIdentityLinking: extResponse.installed_extensions[1].extension.request_identity_link,
                        lastUserIdentityLinkState: false,
                        token: {
                            token: find(
                                extResponse.tokens,
                                (token => token.extension_id === extResponse.installed_extensions[1].extension.id)
                            ).token,
                            permissionsState: 'none',
                            role: 'viewer',
                        },
                    },
                ]);
            });
        });

        test('should skip extensions without valid tokens', function() {
            expect.assertions(1);
            const channelName = `channel_${createRandomStr()}`;
            const channelInfo = api.expectChannelInfo(channelName);
            const extResponse = api.expectOverlayExtensionsForChannel(channelInfo._id, 2);
            // eslint-disable-next-line camelcase
            extResponse.tokens = extResponse.tokens.filter(token => {
                return token.extension_id === extResponse.installed_extensions[1].extension.id;
            });

            return overlayExtensionsForChannel(channelName).then(extConfig => {
                expect(extConfig).toEqual([
                    {
                        id: extResponse.installed_extensions[1].extension.id,
                        name: extResponse.installed_extensions[1].extension.name,
                        sku: extResponse.installed_extensions[1].extension.sku,
                        summary: extResponse.installed_extensions[1].extension.summary,
                        anchor: extResponse.installed_extensions[1].extension.anchor,
                        version: extResponse.installed_extensions[1].extension.version,
                        state: extResponse.installed_extensions[1].extension.state,
                        vendorCode: extResponse.installed_extensions[1].extension.vendor_code,
                        viewerUrl: extResponse.installed_extensions[1].extension.viewer_url,
                        supportsIdentityLinking: extResponse.installed_extensions[1].extension.request_identity_link,
                        lastUserIdentityLinkState: false,
                        token: {
                            token: find(
                                extResponse.tokens,
                                (token => token.extension_id === extResponse.installed_extensions[1].extension.id)
                            ).token,
                            permissionsState: 'none',
                            role: 'viewer',
                        },
                    },
                ]);
            });
        });
    });

    test('postExtensionReport should post data', function() {
        expect.assertions(3);
        const expectedUrl = 'https://www.twitch.tv/user/report';
        const description = [
            'test',
            'Extension ID: 0',
            'Extension Name: bogus',
            'Extension Version: 0',
        ].join('\n');
        const reportData = {
            /* eslint-disable camelcase */
            reported_user: 'twitch',
            authenticity_token: 0,
            tos_ban: false,
            reason: 'other',
            description: description,
            /* eslint-enable camelcase */
        };
        const expectedBody = stringify(reportData);

        api.expectPostExtensionReport();
        return postExtensionReport(reportData).then(() => {
            const request = api.lastCall();
            expect(request[1].method).toBe('POST');
            expect(request[1].body).toBe(expectedBody);
            expect(request[0]).toBe(expectedUrl);
        });
    });

    test('createClip should make a request to given url and return json value', function() {
        expect.assertions(4);
        api.clearFakeResponses();

        const clipCreationUrl = 'fakeurl';
        const oauthTokenResponse = api.setLoggedIn(true);
        const expectedJson = {
            url: 'fakeurl/clipslug',
        };

        api.mock(clipCreationUrl, {
            status: 200,
            body: expectedJson,
        });

        return createClip(clipCreationUrl, {}).then(res => {
            expect(api.called(clipCreationUrl)).toBeTruthy();

            const expectedHeaders = {
                'Content-Type': 'application/x-www-form-urlencoded',
                Authorization: `OAuth ${oauthTokenResponse.token}`,
            };
            const request = api.lastCall(clipCreationUrl);

            expect(request[1].method).toBe('POST');
            expect(request[1].headers).toEqual(expectedHeaders);
            expect(res).toEqual(expectedJson);
        });
    });
});
