import { EmbedHost, PUBLIC_EVENTS } from 'embed/host';
import * as Bridge from 'embed/client';
import { init as initStore } from 'state';
import * as CollectionActions from 'actions/collection';
import { ACTION_TOGGLE_CAPTIONS, setCaptionsData } from 'actions/captions';
import { updateViewerCount } from 'actions/viewercount';
import { resetPlaySession } from 'actions/analytics';
import { setManifestInfo } from 'actions/manifest-info';
import { videoAPILoaded, changeVolume, mutePlayer,
         play, pause } from 'actions/video-api';
import { setOnline } from 'actions/online';
import { setTheatreMode, setFullScreen } from 'actions/screen-mode';
import { ACTION_PLAYING, ACTION_PAUSE, ACTION_ENDED,
         contentIsShowing, updateCurrentTime } from 'actions/playback';
import { embedHostLoaded, ACTION_USE_EMBED_EVENT_EMITTER } from 'actions/embed-event-emitter';
import extend from 'lodash/extend';
import { LiveTwitchContentStream } from 'stream/twitch-live';
import { VODTwitchContentStream } from 'stream/twitch-vod';
import { ProvidedContentStream } from 'stream/provided';
import { ClipContentStream } from 'stream/clip';
import { VideoSourceContentStream } from 'stream/video-source';
import EventEmitter from 'event-emitter';
import { OPEN_STREAM } from 'backend/events/twitch-event';
import { setPlayerType } from 'actions/env';
import { ALL_PLAYER_TYPES, PLAYER_SITE, PLAYER_FRONTPAGE, PLAYER_POPOUT, PLAYER_FEED } from 'util/player-type';
import { TIER_1_EVENTS, VIDEO_PLAYBACK_ERROR } from 'analytics/events';
import Errors from 'errors';
import { ACTION_ERROR } from 'actions/error';
import { ACTION_SET_STREAM } from 'actions/stream';
import { setTrackingProperties } from 'actions/tracking';
import { createRandomStr } from 'test-utils/utils/create-random-string';
import * as api from 'test-utils/utils/api';
import { waitFor } from 'test-utils/utils/waitFor';
import { CLIP_RESPONSE_MINIMUM } from 'test-utils/fixtures/clip';
import difference from 'lodash/difference';

jest.mock('stream/twitch-live', function() {
    return require('../../../mocks/stream/twitch-live');
});

jest.mock('stream/twitch-vod', function() {
    return require('../../../mocks/stream/twitch-vod');
});

jest.mock('stream/provided', function() {
    return require('../../../mocks/stream/provided');
});

const DEFAULT_OPTS = Object.freeze({
    origin: 'https://www.twitch.tv',
});

const TWITCH_EVENTS = Object.freeze([
    Bridge.EVENT_EMBED_OPEN_STREAM,
    Bridge.EVENT_EMBED_VIEWERS_CHANGE,
    Bridge.EVENT_EMBED_CONTENT_SHOWING,
    Bridge.EVENT_THEATRE_ENTERED,
    Bridge.EVENT_THEATRE_EXITED,
    Bridge.EVENT_FULLSCREEN_ENTERED,
    Bridge.EVENT_FULLSCREEN_EXITED,
]);

function buildStoreUpdateObject(store, overrides = {}) {
    const state = store.getState();
    return extend({}, {
        captions: {
            available: state.captions.available,
        },
        viewercount: state.viewercount,
        stats: {
            videoStats: state.stats.videoStats,
        },
        playSessionId: state.analytics.playSessionId,
        broadcastId: state.manifestInfo.broadcast_id,
        screenMode: state.screenMode,
        stream: {
            contentId: state.stream.contentId || '',
            customerId: state.stream.customerId || '',
        },
    }, overrides);
}

function messageWithEvent(event) {
    return {
        namespace: Bridge.BRIDGE_CLIENT_NAMESPACE,
        method: Bridge.BRIDGE_PLAYER_EVENT,
        args: [{ event }],
    };
}

function messageWithEventPayload(event, payload) {
    return {
        namespace: Bridge.BRIDGE_CLIENT_NAMESPACE,
        method: Bridge.BRIDGE_PLAYER_EVENT_WITH_PAYLOAD,
        args: [{
            event,
            data: payload,
        }],
    };
}

const SUBSCRIBE_TO_HOST = Object.freeze({
    args: [],
    method: Bridge.BRIDGE_REQ_SUBSCRIBE,
    namespace: Bridge.BRIDGE_HOST_NAMESPACE,
});

function checkStoreStateSent(postMessageMock, expectedStoreState) {
    const [payload] = postMessageMock;
    expect(payload.method).toBe(Bridge.BRIDGE_STORE_STATE_UPDATE);
    expect(payload.namespace).toBe(Bridge.BRIDGE_CLIENT_NAMESPACE);
    expect(payload.args[0]).toEqual(expectedStoreState);
}

describe('embed | host', () => {
    let store;
    let mockPlayer;
    let windowObj;
    let embedHost;

    beforeEach(() => {
        store = initStore({
            analyticsTracker: {
                trackEvent: jest.fn(),
            },
            window: {
                document: {},
                addEventListener: jest.fn(),
                removeEventListener: jest.fn(),
                postMessage: jest.fn(),
                clearTimeout() {},
                setTimeout() {},
            },
            experiments: {
                get() {
                    return Promise.resolve('');
                },
            },
        });

        const events = new EventEmitter();

        mockPlayer = {
            play: jest.fn(),
            pause: jest.fn(),
            setCurrentTime: jest.fn(),
            setQuality: jest.fn(),
            setVolume: jest.fn(),
            destroy: jest.fn(),
            setMuted: jest.fn(),
            addEventListener(eventName, callback) {
                events.addListener(eventName, callback);
            },
            removeEventListener(eventName, callback) {
                events.removeListener(eventName, callback);
            },
            emit(eventName, payload) {
                events.emit(eventName, payload);
            },
        };

        windowObj = store.getState().window;
        embedHost = new EmbedHost(mockPlayer, store, DEFAULT_OPTS);
        embedHost.handleEvent({
            origin: DEFAULT_OPTS.origin,
            data: SUBSCRIBE_TO_HOST,
            source: windowObj,
        });
        embedHost._sendReadyEvent();
        windowObj.postMessage.mockReset();
        store.dispatch(videoAPILoaded(mockPlayer));
    });

    test('listens to window message event when instantiated', () => {
        expect(windowObj.addEventListener).toBeCalledWith('message', embedHost);
    });

    test('loads embed event emitter when instantiated', () => {
        jest.spyOn(store, 'dispatch');

        store.dispatch(embedHostLoaded(embedHost));

        expect(store.dispatch).toHaveBeenCalledWith({
            type: ACTION_USE_EMBED_EVENT_EMITTER,
            embedHost: embedHost,
        });
    });

    test('unlistens to window when destroyed', () => {
        embedHost.destroy();
        expect(windowObj.removeEventListener).toHaveBeenCalledWith('message', embedHost);
    });

    test('unlistens to player when destroyed', () => {
        jest.spyOn(mockPlayer, 'removeEventListener');
        store.dispatch(setPlayerType(PLAYER_FRONTPAGE));

        mockPlayer.emit(OPEN_STREAM);

        embedHost.destroy();
        expect(mockPlayer.removeEventListener).toHaveBeenCalledTimes(1);
        expect(mockPlayer.removeEventListener.mock.calls[0][0]).toBe(OPEN_STREAM);

        jest.spyOn(embedHost, '_sendPlayerEvent');
        mockPlayer.removeEventListener.mock.calls[0][1]();
        expect(embedHost._sendPlayerEvent).toHaveBeenCalledTimes(1);
        expect(embedHost._sendPlayerEvent.mock.calls[0][0]).toBe(Bridge.EVENT_EMBED_OPEN_STREAM);
    });

    test('on provided content stream change sends store state with content id and customer id', () => {
        const contentId = 'acontentid';
        const customerId = 'acustomerid';

        const expectedClientState = buildStoreUpdateObject(store, {
            stream: {
                contentId,
                customerId,
            },
        });
        store.dispatch({
            type: ACTION_SET_STREAM,
            stream: new ProvidedContentStream({
                customerId,
                contentId,
            }),
        });

        return embedHost._onBridgeReady.then(() => {
            checkStoreStateSent(
                windowObj.postMessage.mock.calls[0],
                expectedClientState
            );
        });
    });

    // eslint-disable-next-line max-len
    test('on non provided content stream change sends store state with empty content id and customer id', () => {
        const expectedClientState = buildStoreUpdateObject(store, {
            stream: {
                contentId: '',
                customerId: '',
            },
        });

        store.dispatch({
            type: ACTION_SET_STREAM,
            stream: new VODTwitchContentStream('a video id'),
        });

        return embedHost._onBridgeReady.then(() => {
            checkStoreStateSent(
                windowObj.postMessage.mock.calls[0],
                expectedClientState
            );
        });
    });

    test('subscribes to contentShowing in store and sends to client', () => {
        windowObj.postMessage.mockReset();

        store.dispatch(contentIsShowing());

        return embedHost._onBridgeReady.then(() => {
            expect(windowObj.postMessage.mock.calls[0][0]).
                toEqual(messageWithEvent(Bridge.EVENT_EMBED_CONTENT_SHOWING));
        });
    });

    test('frontpage player listens for openStream and sends to client', () => {
        store.dispatch(setPlayerType(PLAYER_FRONTPAGE));
        const frontPageEmbedHost = new EmbedHost(mockPlayer, store, DEFAULT_OPTS);
        frontPageEmbedHost.handleEvent({
            origin: DEFAULT_OPTS.origin,
            data: SUBSCRIBE_TO_HOST,
            source: windowObj,
        });

        frontPageEmbedHost._sendReadyEvent();
        windowObj.postMessage.mockReset();

        mockPlayer.emit(OPEN_STREAM);

        return frontPageEmbedHost._onBridgeReady.then(() => {
            expect(windowObj.postMessage.mock.calls[0][0]).
                toEqual(messageWithEvent(Bridge.EVENT_EMBED_OPEN_STREAM));
        });
    });

    test('feed player listens for openStream and sends to client', () => {
        store.dispatch(setPlayerType(PLAYER_FEED));
        const feedPlayerEmbedHost = new EmbedHost(mockPlayer, store, DEFAULT_OPTS);
        feedPlayerEmbedHost.handleEvent({
            origin: DEFAULT_OPTS.origin,
            data: SUBSCRIBE_TO_HOST,
            source: windowObj,
        });
        feedPlayerEmbedHost._sendReadyEvent();
        windowObj.postMessage.mockReset();

        mockPlayer.emit(OPEN_STREAM);

        return feedPlayerEmbedHost._onBridgeReady.then(() => {
            expect(windowObj.postMessage.mock.calls[0][0]).
                toEqual(messageWithEvent(Bridge.EVENT_EMBED_OPEN_STREAM));
        });
    });

    test('on _sendReadyEvent sends ready event with store state', () => {
        embedHost._sendReadyEvent();

        expect(windowObj.postMessage.mock.calls[0][0]).toEqual({
            namespace: Bridge.BRIDGE_CLIENT_NAMESPACE,
            method: Bridge.BRIDGE_HOST_READY,
            args: [buildStoreUpdateObject(store)],
        });
    });

    test('on _sendReadyEvent sends any queued postmessages', () => {
        store.dispatch(contentIsShowing());

        expect(windowObj.postMessage).toHaveBeenCalledTimes(0);

        embedHost._sendReadyEvent();

        expect(windowObj.postMessage.mock.calls[0][0]).toEqual({
            namespace: Bridge.BRIDGE_CLIENT_NAMESPACE,
            method: Bridge.BRIDGE_HOST_READY,
            args: [buildStoreUpdateObject(store)],
        });

        windowObj.postMessage.mockReset();

        return embedHost._onBridgeReady.then(() => {
            expect(windowObj.postMessage.mock.calls[0][0]).
                toEqual(messageWithEvent(Bridge.EVENT_EMBED_CONTENT_SHOWING));
        });
    });

    test('sends wheel event to client', () => {
        embedHost.handleEvent({
            origin: DEFAULT_OPTS.origin,
            data: SUBSCRIBE_TO_HOST,
            source: windowObj,
        });

        windowObj.postMessage.mockReset();

        const event =   {
            deltaMode: 0,
            deltaX: 0,
            deltaY: 0,
            shiftKey: false,
        };
        embedHost._emitWheelEvent(event);

        return embedHost._onBridgeReady.then(() => {
            expect(windowObj.postMessage.mock.calls[0][0]).toEqual({
                namespace: Bridge.BRIDGE_CLIENT_NAMESPACE,
                method: Bridge.BRIDGE_PLAYER_EVENT_WITH_PAYLOAD,
                args: [{
                    event: Bridge.EVENT_EMBED_WHEEL,
                    data: event,
                }],
            });
        });
    });

    test('sends first party analytic event to target origin', () => {
        const origin = 'https://this.is.an.amazon.url';
        const fpaEmbedHost = new EmbedHost(mockPlayer, store, {
            origin,
        });

        fpaEmbedHost.handleEvent({
            origin,
            data: SUBSCRIBE_TO_HOST,
            source: windowObj,
        });
        fpaEmbedHost._sendReadyEvent();
        windowObj.postMessage.mockReset();

        const properties = { exampleProperty: 'exampleProperty' };

        fpaEmbedHost._sendFirstPartyAnalyticEvent(TIER_1_EVENTS.MINUTE_WATCHED, properties);
        fpaEmbedHost._sendFirstPartyAnalyticEvent(TIER_1_EVENTS.VIDEO_PLAY, properties);
        fpaEmbedHost._sendFirstPartyAnalyticEvent(TIER_1_EVENTS.BUFFER_EMPTY, properties);
        fpaEmbedHost._sendFirstPartyAnalyticEvent(VIDEO_PLAYBACK_ERROR, properties);

        return fpaEmbedHost._onBridgeReady.then(() => {
            const postMessageCalls = windowObj.postMessage.mock.calls;
            expect(postMessageCalls[0][0]).
                toEqual(messageWithEventPayload(Bridge.EVENT_EMBED_MINUTE_WATCHED, properties));

            expect(postMessageCalls[0][1]).toBe(origin);

            expect(postMessageCalls[1][0]).
                toEqual(messageWithEventPayload(Bridge.EVENT_EMBED_VIDEO_PLAY, properties));

            expect(postMessageCalls[1][1]).toBe(origin);

            expect(postMessageCalls[2][0]).
                toEqual(messageWithEventPayload(Bridge.EVENT_EMBED_BUFFER_EMPTY, properties));

            expect(postMessageCalls[2][1]).toBe(origin);

            expect(postMessageCalls[3][0]).
                toEqual(messageWithEventPayload(Bridge.EVENT_EMBED_VIDEO_PLAYBACK_ERROR, properties));

            expect(postMessageCalls[3][1]).toBe(origin);
        });
    });

    test('_sendStatsUpdateEvent() sends store state and then emits the statsupdate event', () => {
        const currentStoreState = store.getState();
        const newVideoStats = extend(currentStoreState.stats.videoStats, { fps: 30 });

        const clientState = buildStoreUpdateObject(store, {
            stats: {
                videoStats: newVideoStats,
            },
        });

        embedHost._sendStatsUpdateEvent();

        return embedHost._onBridgeReady.then(() => {
            checkStoreStateSent(
                windowObj.postMessage.mock.calls[0],
                clientState
            );
            expect(windowObj.postMessage.mock.calls[1][0]).
                toEqual(messageWithEvent(Bridge.EVENT_EMBED_STATS_UPDATE));
        });
    });

    test('updating play session id sends store update with playsessionid', () => {
        store.dispatch(resetPlaySession());

        const expectedClientState = buildStoreUpdateObject(store);

        return embedHost._onBridgeReady.then(() => {
            checkStoreStateSent(
                windowObj.postMessage.mock.calls[0],
                expectedClientState
            );
        });
    });

    test('updating broadcast id in manifest info sends store update', () => {
        store.dispatch(setManifestInfo({
            // eslint-disable-next-line camelcase
            broadcast_id: createRandomStr(),
        }));

        const expectedClientState = buildStoreUpdateObject(store);

        return embedHost._onBridgeReady.then(() => {
            checkStoreStateSent(
                windowObj.postMessage.mock.calls[0],
                expectedClientState
            );
        });
    });

    test('subscribes to viewercount in state store and sends to client', () => {
        const newViewerCount = 50;
        store.dispatch(updateViewerCount(newViewerCount));
        const clientState = buildStoreUpdateObject(store);

        // eslint-disable-next-line max-len
        return embedHost._onBridgeReady.then(() => {
            checkStoreStateSent(
                windowObj.postMessage.mock.calls[0],
                clientState
            );
            expect(windowObj.postMessage.mock.calls[1][0]).
                toEqual(messageWithEvent(Bridge.EVENT_EMBED_VIEWERS_CHANGE));
        });
    });

    test('subscribes to online in state store and sends appropriate embed events', () => {
        expect(store.getState().online).toBe(false);
        store.dispatch(setOnline(true));
        store.dispatch(setOnline(false));

        return embedHost._onBridgeReady.then(() => {
            expect(windowObj.postMessage.mock.calls[0][0]).
                toEqual(messageWithEvent(Bridge.EVENT_EMBED_ONLINE));

            expect(windowObj.postMessage.mock.calls[1][0]).
                toEqual(messageWithEvent(Bridge.EVENT_EMBED_OFFLINE));
        });
    });

    test('subscribes to theatre mode in state store and sends appropriate embed events', () => {
        expect(store.getState().screenMode.isTheatreMode).toBe(false);
        store.dispatch(setTheatreMode(true));
        const clientState1 = buildStoreUpdateObject(store);
        store.dispatch(setTheatreMode(false));
        const clientState2 = buildStoreUpdateObject(store);

        return embedHost._onBridgeReady.then(() => {
            const postMessageCalls = windowObj.postMessage.mock.calls;
            checkStoreStateSent(postMessageCalls[0], clientState1);
            expect(postMessageCalls[1][0]).toEqual(messageWithEvent(Bridge.EVENT_THEATRE_ENTERED));
            checkStoreStateSent(postMessageCalls[2], clientState2);
            expect(postMessageCalls[3][0]).toEqual(messageWithEvent(Bridge.EVENT_THEATRE_EXITED));
        });
    });

    test('subscribes to fullscreen mode in state store and sends appropriate embed events', () => {
        expect(store.getState().screenMode.isFullScreen).toBe(false);
        store.dispatch(setFullScreen(true));
        const clientState1 = buildStoreUpdateObject(store);
        store.dispatch(setFullScreen(false));
        const clientState2 = buildStoreUpdateObject(store);

        return embedHost._onBridgeReady.then(() => {
            const postMessageCalls = windowObj.postMessage.mock.calls;

            checkStoreStateSent(postMessageCalls[0], clientState1);

            expect(postMessageCalls[1][0]).
                toEqual(messageWithEvent(Bridge.EVENT_FULLSCREEN_ENTERED));

            checkStoreStateSent(postMessageCalls[2], clientState2);

            expect(postMessageCalls[3][0]).
                toEqual(messageWithEvent(Bridge.EVENT_FULLSCREEN_EXITED));
        });
    });

    test('subscribes to captions availability and sends state update on changes', () => {
        const fakeCaptionsData = {
            data: ['a', 'caption', 'string'],
        };

        store.dispatch(setCaptionsData(fakeCaptionsData));
        const clientState = buildStoreUpdateObject(store);

        return embedHost._onBridgeReady.then(() => {
            const postMessageCalls = windowObj.postMessage.mock.calls;
            checkStoreStateSent(postMessageCalls[0], clientState);
        });
    });

    test('subscribes to ended and paused in state store and sends appropriate embed events', () => {
        expect(store.getState().playback.ended).toBe(false);
        expect(store.getState().playback.paused).toBe(false);

        store.dispatch({ type: ACTION_PLAYING });
        store.dispatch({ type: ACTION_PAUSE });
        store.dispatch({ type: ACTION_ENDED });

        return embedHost._onBridgeReady.then(() => {
            const postMessageCalls = windowObj.postMessage.mock.calls;
            expect(postMessageCalls[1][0]).toEqual(messageWithEvent(Bridge.EVENT_EMBED_PLAY));
            expect(postMessageCalls[2][0]).toEqual(messageWithEvent(Bridge.EVENT_EMBED_PAUSE));
            expect(postMessageCalls[4][0]).toEqual(messageWithEvent(Bridge.EVENT_EMBED_ENDED));
        });
    });

    test('subscribes to currentTime in state store and sends appropriate embed events', () => {
        expect(store.getState().playback.currentTime).toBe(0);
        const currentTime = 99;
        store.dispatch(updateCurrentTime(currentTime));

        return embedHost._onBridgeReady.then(() => {
            expect(windowObj.postMessage.mock.calls[0][0]).
                toEqual(messageWithEventPayload(Bridge.EVENT_EMBED_TIME_UPDATE, currentTime));
        });
    });

    test('checks error in state store on instantiation and queues appropriate embed events ', () => {
        const errorCode = Errors.CODES.ABORTED;
        store.dispatch({
            type: ACTION_ERROR,
            code: errorCode,
        });

        const errorEmbedHost = new EmbedHost(mockPlayer, store, DEFAULT_OPTS);

        errorEmbedHost.handleEvent({
            origin: DEFAULT_OPTS.origin,
            data: SUBSCRIBE_TO_HOST,
            source: windowObj,
        });

        errorEmbedHost._sendReadyEvent();

        windowObj.postMessage.mockReset();
        expect(windowObj.postMessage).toHaveBeenCalledTimes(0);

        return errorEmbedHost._onBridgeReady.then(() => {
            expect(windowObj.postMessage.mock.calls[0][0]).
                toEqual(messageWithEventPayload(Bridge.EVENT_EMBED_ERROR, { code: errorCode }));
        });
    });

    test('subscribes to error in state store and sends appropriate embed event', () => {
        const errorCode = Errors.CODES.ABORTED;

        store.dispatch({
            type: ACTION_ERROR,
            code: errorCode,
        });

        return embedHost._onBridgeReady.then(() => {
            expect(windowObj.postMessage.mock.calls[0][0]).
                toEqual(messageWithEventPayload(Bridge.EVENT_EMBED_ERROR, { code: errorCode }));
        });
    });

    test('sends appropriate seeked embed events', () => {
        const seekTimestamp = 99;
        embedHost._sendSeekedEvent(99);

        return embedHost._onBridgeReady.then(() => {
            expect(windowObj.postMessage.mock.calls[0][0]).
                toEqual(messageWithEventPayload(Bridge.EVENT_EMBED_SEEKED, seekTimestamp));
        });
    });

    describe('on state update', () => {
        test('sends player event to client', () => {
            store.dispatch({
                type: ACTION_PLAYING,
            });

            return embedHost._onBridgeReady.then(() => {
                expect(windowObj.postMessage.mock.calls[1][0]).
                    toEqual(messageWithEvent(Bridge.EVENT_EMBED_PLAY));
            });
        });
    });

    describe('only posts to public origins on public events', () => {
        function testPublicEvents(eventName) {
            const isPublic = PUBLIC_EVENTS[eventName];
            const NOT_DEFAULT_OPTS = { origin: 'https://www.nottwitch.tv' };
            const testMsg = isPublic ?
                `posts public event ${eventName} to public origin ${NOT_DEFAULT_OPTS.origin}` :
                `does not post ${eventName} to ${NOT_DEFAULT_OPTS.origin}`;

            test(testMsg, () => {
                const publicEmbedHost = new EmbedHost(mockPlayer, store, NOT_DEFAULT_OPTS);
                publicEmbedHost.handleEvent({
                    origin: NOT_DEFAULT_OPTS.origin,
                    data: SUBSCRIBE_TO_HOST,
                    source: windowObj,
                });

                publicEmbedHost._sendReadyEvent();
                windowObj.postMessage.mockReset();
                publicEmbedHost._sendPlayerEvent(eventName);

                return publicEmbedHost._onBridgeReady.then(() => {
                    if (!isPublic) {
                        expect(windowObj.postMessage).toHaveBeenCalledTimes(0);
                        return;
                    }

                    const [, targetOrigin] = windowObj.postMessage.mock.calls[0];
                    expect(targetOrigin).toBe(NOT_DEFAULT_OPTS.origin);
                });
            });
        }

        Object.keys(PUBLIC_EVENTS).forEach(testPublicEvents);
        TWITCH_EVENTS.forEach(testPublicEvents);

        test('should post to 127.0.0.1', () => {
            const origin = 'http://127.0.0.1:21312';
            const desktopEmbedHost = new EmbedHost(mockPlayer, store, {
                origin,
            });

            desktopEmbedHost.handleEvent({
                origin,
                data: SUBSCRIBE_TO_HOST,
                source: windowObj,
            });

            desktopEmbedHost._sendReadyEvent();

            windowObj.postMessage.mockReset();
            desktopEmbedHost._sendPlayerEvent(TWITCH_EVENTS[1]);

            return desktopEmbedHost._onBridgeReady.then(() => {
                expect(windowObj.postMessage).toHaveBeenCalledTimes(1);
                const [, targetOrigin] = windowObj.postMessage.mock.calls[0];

                expect(targetOrigin).toBe(origin);
            });
        });
    });

    describe('calls state method', () => {
        const channelId = '27446517';
        const channelName = 'monstercat';
        const videoID = 'v456345';
        const timestamp = 12345;
        const contentId = 'fakecontentid';
        const customerId = 'fakecustomerid';

        beforeEach(() => {
            jest.spyOn(store, 'dispatch');

            api.setLoggedIn(true);
            api.expectChannelInfo(channelName);
            api.expectChannelInfoByChannelId(channelId, channelName);
            api.expectStreamInfo(channelName);

            embedHost.handleEvent({
                origin: DEFAULT_OPTS.origin,
                data: SUBSCRIBE_TO_HOST,
                source: windowObj,
            });
        });

        afterEach(() => {
            api.clearFakeResponses();
        });

        test('setChannel', () => {
            const channelName = 'monstercat';
            const setChannelMessage = {
                args: [channelName],
                method: Bridge.METHOD_SET_CHANNEL,
                namespace: Bridge.BRIDGE_HOST_NAMESPACE,
            };

            embedHost.handleEvent({
                origin: DEFAULT_OPTS.origin,
                data: setChannelMessage,
                source: windowObj,
            });

            const { stream, streamMetadata } = store.getState();

            expect(stream instanceof LiveTwitchContentStream).toBe(true);
            expect(stream.channel).toBe(channelName);
            expect(streamMetadata.channelName).toBe(channelName);
        });

        test('setChannelId', () => {
            const setChannelIdMessage = {
                args: [channelId],
                method: Bridge.METHOD_SET_CHANNEL_ID,
                namespace: Bridge.BRIDGE_HOST_NAMESPACE,
            };
            embedHost.handleEvent({
                origin: DEFAULT_OPTS.origin,
                data: setChannelIdMessage,
                source: windowObj,
            });

            return waitFor(() => api.called()).then(() => {
                const { stream , streamMetadata } = store.getState();
                expect(stream instanceof LiveTwitchContentStream).toBe(true);
                expect(stream.channel).toBe(channelName);
                expect(streamMetadata.channelName).toBe(channelName);
            });
        });

        test('setVideo', () => {
            api.expectVideoInfo(videoID);
            const setVideoMessage = {
                args: [videoID, timestamp],
                method: Bridge.METHOD_SET_VIDEO,
                namespace: Bridge.BRIDGE_HOST_NAMESPACE,
            };

            embedHost.handleEvent({
                origin: DEFAULT_OPTS.origin,
                data: setVideoMessage,
                source: windowObj,
            });

            const { stream, streamMetadata, playback } = store.getState();

            return waitFor(() => api.called()).then(() => {
                expect(stream instanceof VODTwitchContentStream).toBe(true);
                expect(stream.videoId).toBe(videoID);
                expect(streamMetadata.videoId).toBe(videoID);
                expect(playback.startTime).toBe(timestamp);
            });
        });

        test('enableCaptions', () => {
            const enableCaptionsMessage = {
                args: [],
                method: Bridge.METHOD_ENABLE_CAPTIONS,
                namespace: Bridge.BRIDGE_HOST_NAMESPACE,
            };

            store.dispatch.mockReset();

            embedHost.handleEvent({
                origin: DEFAULT_OPTS.origin,
                data: enableCaptionsMessage,
                source: windowObj,
            });

            const thunk = store.dispatch.mock.calls[0][0];
            const spy = jest.fn();

            thunk(spy, () => ({
                captions: {},
            }));

            expect(spy.mock.calls[0][0].type).toBe(ACTION_TOGGLE_CAPTIONS);
            expect(spy.mock.calls[0][0].captions.enabled).toBe(true);
        });

        test('disableCaptions', () => {
            const enableCaptionsMessage = {
                args: [],
                method: Bridge.METHOD_DISABLE_CAPTIONS,
                namespace: Bridge.BRIDGE_HOST_NAMESPACE,
            };

            store.dispatch.mockReset();

            embedHost.handleEvent({
                origin: DEFAULT_OPTS.origin,
                data: enableCaptionsMessage,
                source: windowObj,
            });

            const thunk = store.dispatch.mock.calls[0][0];
            const spy = jest.fn();

            thunk(spy, () => ({
                captions: {},
            }));
            expect(spy.mock.calls[0][0].type).toBe(ACTION_TOGGLE_CAPTIONS);
            expect(spy.mock.calls[0][0].captions.enabled).toBe(false);
        });

        test('setCaptionSize', () => {
            store.dispatch.mockReset();

            embedHost.handleEvent({
                origin: DEFAULT_OPTS.origin,
                data: {
                    method: Bridge.METHOD_SET_CAPTION_SIZE,
                    args: [33],
                    namespace: Bridge.BRIDGE_HOST_NAMESPACE,
                },
                source: windowObj,
            });

            const thunk = store.dispatch.mock.calls[0][0];
            const dispatch = jest.fn();

            thunk(dispatch, store.getState);

            expect(dispatch.mock.calls[0][0].captions.style.fontSize).toBe(33);
        });

        test('setContent', () => {
            const setContentMessage = {
                args: [contentId, customerId],
                method: Bridge.METHOD_SET_CONTENT,
                namespace: Bridge.BRIDGE_HOST_NAMESPACE,
            };

            embedHost.handleEvent({
                origin: DEFAULT_OPTS.origin,
                data: setContentMessage,
                source: windowObj,
            });

            const { stream } = store.getState();

            expect(stream instanceof ProvidedContentStream).toBe(true);
            expect(stream.contentId).toBe(contentId);
            expect(stream.customerId).toBe(customerId);
        });

        test('setClip', () => {
            const slug = 'SomeSlug';
            api.expectClipStatus(slug, {
                // eslint-disable-next-line camelcase
                quality_options: [],
            });
            api.expectClipInfo(CLIP_RESPONSE_MINIMUM);
            api.expectClipView(slug, {});

            const setClipMessage = {
                args: [slug],
                method: Bridge.METHOD_SET_CLIP,
                namespace: Bridge.BRIDGE_HOST_NAMESPACE,
            };

            embedHost.handleEvent({
                origin: DEFAULT_OPTS.origin,
                data: setClipMessage,
                source: windowObj,
            });

            // wait for at least the first clip api call to complete
            return waitFor(() => api.calls().matched.length > 1).then(() => {
                const { stream } = store.getState();
                expect(stream instanceof ClipContentStream).toBe(true);
                expect(stream.slug).toBe(slug);
            });
        });

        test('setVideoSource', () => {
            const inputUrl = 'https://twitch.tv/some.mp4';

            const setVideoSourceMessage = {
                args: [inputUrl],
                method: Bridge.METHOD_SET_VIDEO_SOURCE,
                namespace: Bridge.BRIDGE_HOST_NAMESPACE,
            };

            embedHost.handleEvent({
                origin: DEFAULT_OPTS.origin,
                data: setVideoSourceMessage,
                source: windowObj,
            });

            const { stream } = store.getState();
            expect(stream instanceof VideoSourceContentStream).toBe(true);

            return stream.streamUrl.then(url => {
                expect(inputUrl).toBe(url);
            });
        });

        describe('setCollection', () => {
            let collectionId;
            let setCollectionMessage;

            beforeEach(() => {
                collectionId = createRandomStr();
                setCollectionMessage = {
                    args: [collectionId],
                    method: Bridge.METHOD_SET_COLLECTION,
                    namespace: Bridge.BRIDGE_HOST_NAMESPACE,
                };
            });

            test('setting a collection with no start video', () => {
                store.dispatch.mockReset();

                embedHost.handleEvent({
                    origin: DEFAULT_OPTS.origin,
                    data: setCollectionMessage,
                    source: windowObj,
                });

                return waitFor(() => store.dispatch.mock.calls.length).then(() => {
                    const [requestCollectionAction] = store.dispatch.mock.calls[0];
                    expect(requestCollectionAction).toEqual({
                        type: CollectionActions.ACTION_REQUEST_COLLECTION,
                        request: {
                            collectionId,
                            videoId: '',
                            timestamp: '',
                            preferVideo: false,
                        },
                    });
                });
            });

            test('setting a collection with a video', () => {
                const fakeVideoId = 'v123123';
                setCollectionMessage.args.push(fakeVideoId);

                store.dispatch.mockReset();

                embedHost.handleEvent({
                    origin: DEFAULT_OPTS.origin,
                    data: setCollectionMessage,
                    source: windowObj,
                });

                return waitFor(() => store.dispatch.mock.calls.length).then(() => {
                    const [requestCollectionAction] = store.dispatch.mock.calls[0];
                    expect(requestCollectionAction).toEqual({
                        type: CollectionActions.ACTION_REQUEST_COLLECTION,
                        request: {
                            collectionId,
                            videoId: fakeVideoId,
                            timestamp: '',
                            preferVideo: false,
                        },
                    });
                });
            });

            test('setting a collection with a video and a timestamp', () => {
                const fakeVideoId = 'v123123';
                const timestamp = 12345;
                setCollectionMessage.args.push(fakeVideoId);
                setCollectionMessage.args.push(timestamp);

                store.dispatch.mockReset();

                embedHost.handleEvent({
                    origin: DEFAULT_OPTS.origin,
                    data: setCollectionMessage,
                    source: windowObj,
                });

                return waitFor(() => store.dispatch.mock.calls.length).then(() => {
                    const [requestCollectionAction] = store.dispatch.mock.calls[0];
                    expect(requestCollectionAction).toEqual({
                        type: CollectionActions.ACTION_REQUEST_COLLECTION,
                        request: {
                            collectionId,
                            videoId: fakeVideoId,
                            timestamp,
                            preferVideo: false,
                        },
                    });
                });
            });
        });

        test('setQuality', () => {
            const qualityName = 'source';
            const setQualityMessage = {
                args: [qualityName],
                method: Bridge.METHOD_SET_QUALITY,
                namespace: Bridge.BRIDGE_HOST_NAMESPACE,
            };

            embedHost.handleEvent({
                origin: DEFAULT_OPTS.origin,
                data: setQualityMessage,
                source: windowObj,
            });

            const { quality } = store.getState();

            expect(quality.selected).toBe(qualityName);
        });

        test('setMiniPlayerMode', () => {
            store.dispatch.mockRestore();

            const setMiniPlayerMode = {
                args: [true],
                method: Bridge.METHOD_SET_MINI_PLAYER_MODE,
                namespace: Bridge.BRIDGE_HOST_NAMESPACE,
            };

            embedHost.handleEvent({
                origin: DEFAULT_OPTS.origin,
                data: setMiniPlayerMode,
                source: windowObj,
            });

            const { ui } = store.getState();
            expect(ui.isMini).toBe(true);
        });
    });

    describe('calls player method', () => {
        beforeEach(() => {
            jest.spyOn(store, 'dispatch');
        });

        test('play', () => {
            const playMessage = {
                args: [],
                method: Bridge.METHOD_PLAY,
                namespace: Bridge.BRIDGE_HOST_NAMESPACE,
            };

            embedHost.handleEvent({
                origin: DEFAULT_OPTS.origin,
                data: playMessage,
                source: windowObj,
            });

            expect(store.dispatch).toHaveBeenCalledTimes(1);
            expect(store.dispatch).toBeCalledWith(play());
        });

        test('pause', () => {
            const pauseMessage = {
                args: [],
                method: Bridge.METHOD_PAUSE,
                namespace: Bridge.BRIDGE_HOST_NAMESPACE,
            };

            embedHost.handleEvent({
                origin: DEFAULT_OPTS.origin,
                data: pauseMessage,
                source: windowObj,
            });

            expect(store.dispatch).toHaveBeenCalledTimes(1);
            expect(store.dispatch).toHaveBeenCalledWith(pause());
        });

        test('seek', () => {
            const currentTime = 100;
            const seekMessage = {
                args: [currentTime],
                method: Bridge.METHOD_SEEK,
                namespace: Bridge.BRIDGE_HOST_NAMESPACE,
            };

            embedHost.handleEvent({
                origin: DEFAULT_OPTS.origin,
                data: seekMessage,
                source: windowObj,
            });

            expect(mockPlayer.setCurrentTime).toHaveBeenCalledTimes(1);
            expect(mockPlayer.setCurrentTime).toHaveBeenCalledWith(parseFloat(currentTime));
        });

        test('setVolume', () => {
            const volume = 0.2;
            const setVolumeMessage = {
                args: [volume],
                method: Bridge.METHOD_SET_VOLUME,
                namespace: Bridge.BRIDGE_HOST_NAMESPACE,
            };

            embedHost.handleEvent({
                origin: DEFAULT_OPTS.origin,
                data: setVolumeMessage,
                source: windowObj,
            });

            expect(store.dispatch).toHaveBeenCalledTimes(1);
            expect(store.dispatch).toHaveBeenCalledWith(changeVolume(volume));
        });

        test('destroy', () => {
            const destroyMessage = {
                args: [],
                method: Bridge.METHOD_DESTROY,
                namespace: Bridge.BRIDGE_HOST_NAMESPACE,
            };

            jest.spyOn(embedHost, 'destroy');
            jest.spyOn(embedHost, '_sendAll');

            embedHost.handleEvent({
                origin: DEFAULT_OPTS.origin,
                data: destroyMessage,
                source: windowObj,
            });

            expect(mockPlayer.destroy).toHaveBeenCalledTimes(1);
            expect(embedHost.destroy).toHaveBeenCalledTimes(1);
            expect(embedHost._sendAll).toHaveBeenCalledWith(Bridge.BRIDGE_DESTROY);
        });

        test('mute', () => {
            const mute = true;
            const setMuteMessage = {
                args: [mute],
                method: Bridge.METHOD_SET_MUTE,
                namespace: Bridge.BRIDGE_HOST_NAMESPACE,
            };

            embedHost.handleEvent({
                origin: DEFAULT_OPTS.origin,
                data: setMuteMessage,
                source: windowObj,
            });

            expect(store.dispatch).toHaveBeenCalledTimes(1);
            expect(store.dispatch.mock.calls[0][0]).toEqual(mutePlayer(mute));
        });

        test('setPlayerType', () => {
            const playerType = 'site';
            const setPlayerTypeMessage = {
                args: [playerType],
                method: Bridge.METHOD_SET_PLAYER_TYPE,
                namespace: Bridge.BRIDGE_HOST_NAMESPACE,
            };

            embedHost.handleEvent({
                origin: DEFAULT_OPTS.origin,
                data: setPlayerTypeMessage,
                source: windowObj,
            });

            expect(store.dispatch).toHaveBeenCalledTimes(1);
            expect(store.dispatch.mock.calls[0][0]).toEqual(setPlayerType(playerType));
        });

        describe('setTrackingProperties', () => {
            let newProps;

            beforeEach(() => {
                jest.spyOn(store, 'dispatch');
                newProps = {
                    /* eslint-disable camelcase */
                    app_session_id: 'appsessionid',
                    benchmark_session_id: 'benchmarksessionid',
                    client_app: 'newclientapp',
                    medium: 'newmedium',
                    content: 'newcontent',
                    player: 'newplayer',
                    referrer: 'newreferrer',
                    referrer_url: 'newreferrerurl',
                    /* eslint-enable camelcase */
                };
            });

            const settableTypes = [
                PLAYER_SITE,
                PLAYER_POPOUT,
                PLAYER_FRONTPAGE,
                PLAYER_FEED,
            ];

            const unsettableTypes = difference(ALL_PLAYER_TYPES, settableTypes);

            function testSetTrackingProperties(playerType) {
                test(`sets tracking properties if ${playerType} player`, () => {
                    const setTrackingPropertiesMessage = {
                        args: [newProps],
                        method: Bridge.METHOD_SET_TRACKING_PROPERTIES,
                        namespace: Bridge.BRIDGE_HOST_NAMESPACE,
                    };

                    store.dispatch(setPlayerType(playerType));
                    store.dispatch.mockReset();

                    embedHost.handleEvent({
                        origin: DEFAULT_OPTS.origin,
                        data: setTrackingPropertiesMessage,
                        source: windowObj,
                    });

                    expect(store.dispatch).toHaveBeenCalledTimes(1);
                    expect(store.dispatch.mock.calls[0][0]).toEqual(setTrackingProperties(newProps));
                });
            }

            settableTypes.forEach(function(type) {
                testSetTrackingProperties(type);
            });

            function testCannotSetTrackingProperties(playerType) {
                test(`does not set tracking properties for ${playerType} player`, () => {
                    const setTrackingPropertiesMessage = {
                        args: [newProps],
                        method: Bridge.METHOD_SET_TRACKING_PROPERTIES,
                        namespace: Bridge.BRIDGE_HOST_NAMESPACE,
                    };

                    store.dispatch(setPlayerType(playerType));
                    store.dispatch.mockReset();

                    embedHost.handleEvent({
                        origin: DEFAULT_OPTS.origin,
                        data: setTrackingPropertiesMessage,
                        source: windowObj,
                    });

                    expect(store.dispatch).toHaveBeenCalledTimes(0);
                });
            }

            unsettableTypes.forEach(function(type) {
                testCannotSetTrackingProperties(type);
            });
        });
    });
});
