import { init as initStore } from 'state';
import assign from 'lodash/assign';
import isFunction from 'lodash/isFunction';
import { FollowManager } from 'follow-manager';
import { setOnline } from 'actions/online';
import { ACTION_SET_STREAMMETADATA } from 'actions/stream-metadata';
import { ACTION_SET_STREAM } from 'actions/stream';
import { ACTION_PLAYING, ACTION_ENDED } from 'actions/playback';
import { userLoggedIn } from 'actions/user';
import { followInfoFetched } from 'actions/follow';
import { waitFor } from 'test-utils/utils/waitFor';
import { OFFLINE_STATUS, UNKNOWN_STATUS } from 'state/online-status';
import { PLATFORM_MOBILE_WEB } from 'state/env';
import { createRandomStr } from 'test-utils/utils/create-random-string';
import * as api from 'test-utils/utils/api';

describe('follow-manager', () => {
    let store;
    let followManager;

    beforeEach(() => {
        store = initStore();
        jest.spyOn(store, 'dispatch');
        followManager = new FollowManager(store);
    });

    test('should not dispatch any action if channel is blacklisted', () => {
        store.dispatch({
            type: ACTION_SET_STREAM,
            stream: {
                contentType: 'live',
                channel: 'aMaZon_LivesTream',
            },
        });
        store.dispatch.mockReset();

        followManager._fetchFollowInfo();

        expect(store.dispatch).toHaveBeenCalledTimes(0);
    });

    test('should fire _fetchFollowInfo if offline and playback.hasPlayed is false', () => {
        followManager._fetchFollowInfo = jest.fn();
        store.dispatch(setOnline(false));

        expect(store.getState().onlineStatus).toBe(OFFLINE_STATUS);
        expect(store.getState().playback.hasPlayed).toBe(false);
        expect(followManager._fetchFollowInfo.mock.calls.length).toBe(1);
    });

    test('should not fire _fetchFollowInfo if offline and playback.hasPlayed is true', () => {
        followManager._fetchFollowInfo = jest.fn();

        // Mimic content playing
        store.dispatch(setOnline(true));
        store.dispatch({
            type: ACTION_PLAYING,
        });

        // Mimic broadcast going offline, but content is still playing
        store.dispatch(setOnline(false));

        expect(store.getState().onlineStatus).toBe(OFFLINE_STATUS);
        expect(store.getState().playback.hasPlayed).toBe(true);
        expect(followManager._fetchFollowInfo).toHaveBeenCalledTimes(0);
    });

    test('should not fire _fetchFollowInfo if onlineStatus is not offline (VOD)', () => {
        followManager._fetchFollowInfo = jest.fn();

        // Mimic content playing
        store.dispatch({
            type: ACTION_PLAYING,
        });
        expect(store.getState().playback.hasPlayed).toBe(true);

        // Mimic content ending
        store.dispatch({
            type: ACTION_ENDED,
        });
        expect(store.getState().onlineStatus).toBe(UNKNOWN_STATUS);
        expect(followManager._fetchFollowInfo).toHaveBeenCalledTimes(0);
    });

    test('should fire _fetchFollowInfo if offline, hasPlayed, and ended', () => {
        followManager._fetchFollowInfo = jest.fn();

        // Mimic content playing
        store.dispatch(setOnline(true));
        store.dispatch({
            type: ACTION_PLAYING,
        });

        // Mimic broadcast ending
        store.dispatch(setOnline(false));
        store.dispatch({
            type: ACTION_ENDED,
        });

        expect(store.getState().onlineStatus).toBe(OFFLINE_STATUS);
        expect(store.getState().playback.hasPlayed).toBe(true);
        expect(store.getState().playback.ended).toBe(true);

        expect(followManager._fetchFollowInfo.mock.calls.length).toBe(1);
    });

    test('_getIDs returns channelId, userId, and logged in status', () => {
        const channelId = 123461234;
        const userId = 123142123;

        store.dispatch({
            type: ACTION_SET_STREAMMETADATA,
            streamMetadata: {
                channel: {
                    id: channelId,
                },
            },
        });

        store.dispatch(userLoggedIn({
            id: userId,
        }));

        expect(followManager._getIDs()).toEqual({
            channelId,
            userId,
            loggedIn: true,
        });
    });

    test('destroy unsubs all items on store', () => {
        followManager._fetchFollowInfo = jest.fn();
        followManager.destroy();
        store.dispatch(setOnline(false));
        expect(followManager._fetchFollowInfo).toHaveBeenCalledTimes(0);
    });

    describe('_fetchFollowInfo', () => {
        let setState;

        beforeEach(() => {
            setState = state => {
                const DEFAULT = initStore().getState();
                store.getState = jest.fn().mockReturnValue(assign({}, DEFAULT, state));
            };
        });

        test('should no-op if user is own their own channel', () => {
            const sameId = createRandomStr();

            followManager._getIDs = jest.fn().mockReturnValue({
                userId: sameId,
                channelId: sameId,
                loggedIn: true,
            });

            followManager._fetchFollowInfo();
            expect(store.dispatch).toHaveBeenCalledTimes(0);
        });

        test('should no-op if no channelId available', () => {
            const userId = createRandomStr();

            followManager._getIDs = jest.fn().mockReturnValue({
                userId,
                channelId: null,
                loggedIn: true,
            });

            followManager._fetchFollowInfo();
            expect(store.dispatch).toHaveBeenCalledTimes(0);
        });

        test('should no-op if on mobile_web', () => {
            const userId = createRandomStr();
            const channelId = parseInt(Math.random() * 1000, 10);
            setState({
                env: {
                    platform: PLATFORM_MOBILE_WEB,
                },
            });

            followManager._getIDs = jest.fn().mockReturnValue({
                userId,
                channelId,
                loggedIn: true,
            });

            followManager._fetchFollowInfo();
            expect(store.dispatch).toHaveBeenCalledTimes(0);
        });

        test('should immediately update store if user is logged off', () => {
            const channelId = createRandomStr();

            followManager._getIDs = jest.fn().mockReturnValue({
                channelId,
                userId: null,
                loggedIn: false,
            });

            followManager._fetchFollowInfo();
            expect(store.dispatch).toHaveBeenCalledTimes(1);
            expect(store.dispatch.mock.calls[0][0]).toEqual(followInfoFetched({
                following: false,
                notificationsEnabled: false,
            }));
        });

        test('should dispatch fetchFollowInfo thunk if user is logged in', () => {
            const userId = 12351223413;
            const channelId = 123123252113;
            const stream = {};

            api.setLoggedIn(true);

            api.expectFollowInfo({
                userId,
                channelId,
                following: false,
                notificationsEnabled: false,
            });

            followManager._getIDs = jest.fn().mockReturnValue({
                channelId,
                userId,
                loggedIn: true,
            });

            followManager._fetchFollowInfo();
            expect(store.dispatch).toHaveBeenCalledTimes(1);

            const thunk = store.dispatch.mock.calls[0][0];
            expect(isFunction(thunk)).toBeTruthy();

            const dispatchSpy = jest.fn();
            thunk(dispatchSpy, () => ({ stream }));

            return waitFor(() => dispatchSpy.mock.calls.length).
                then(() => {
                    expect(dispatchSpy.mock.calls[0][0]).toEqual(followInfoFetched({
                        following: false,
                        notificationsEnabled: false,
                    }));
                });
        });
    });
});
