import assign from 'lodash/assign';
import { createUIMiddleware } from 'middleware/ui-middleware';
import * as UIActions from 'actions/ui';
import * as PlaybackActions from 'actions/playback';
import * as ScreenActions from 'actions/screen-mode';
import { ACTION_SET_CASTING_STATE } from 'actions/chromecast';
import { ONLINE_STATUS, OFFLINE_STATUS } from 'state/online-status';
import { SIDEBAR_VIEW } from 'state/collection';
import { ACTION_SET_CURRENT_QUALITY } from 'actions/quality';
import { PLAYER_HIGHLIGHTER } from 'util/player-type';
import { buildFakeWindow } from 'test-utils/fakes/window.fake';
import { initialControlsDelay, hoverControlsDelay } from 'settings';
import isFunction from 'lodash/isFunction';

describe('middleware | ui-middleware', function() {
    let setupMiddleware;
    let dispatch;
    let getState;
    let next;
    let dispatchAction;

    beforeEach(function() {
        setupMiddleware = overrides => {
            dispatch = jest.fn();
            const windowObj = buildFakeWindow();
            getState = () => {
                return assign(
                    {},
                    {
                        collection: {
                            currentView: '',
                        },
                        env: {
                            playerType: '',
                        },
                        onlineStatus: ONLINE_STATUS,
                        playback: {
                            paused: false,
                        },
                        ui: {
                            isMini: false,
                            onUIElement: false,
                        },
                        window: windowObj,
                        quality: {},
                    },
                    overrides
                );
            };
            const middleware = createUIMiddleware();
            next = middleware({
                dispatch,
                getState,
            });
            dispatchAction = next(() => {});
        };

        setupMiddleware();
    });

    test('it must return a function to handle the action', function() {
        const actionHandler = next();
        expect(isFunction(actionHandler)).toBeTruthy();
    });

    test('it must pass the action into the next callback', function() {
        const nextSpy = jest.fn();
        const mockAction = { type: 'mock action' };
        const actionHandler = next(nextSpy);
        actionHandler(mockAction);

        const [dispatchedAction] = nextSpy.mock.calls[0];
        expect(dispatchedAction).toEqual(mockAction);
    });

    describe('on creation', function() {
        test('should call showUIWithAutoHide', function() {
            const action = UIActions.showUIWithAutoHide();
            expect(dispatch.mock.calls[0][0]).toEqual(action);
        });

        test('should call setTimeout on window with proper args', function() {
            expect(getState().window.setTimeout).toHaveBeenCalledTimes(1);
            expect(getState().window.setTimeout.mock.calls[0][1]).toBe(initialControlsDelay);
        });
    });

    describe('hideUICallback', function() {
        // eslint-disable-next-line max-len
        test('should call setUIDisplay with false if not in highlighter, not on element and not paused', function() {
            const action = UIActions.setUIDisplay(false);
            expect(dispatch.mock.calls[1][0]).toEqual(action);
        });

        test('should call not setUIDisplay if in highlighter', function() {
            setupMiddleware({
                env: {
                    playerType: PLAYER_HIGHLIGHTER,
                },
            });
            expect(dispatch).toHaveBeenCalledTimes(1);
        });

        test('should call not setUIDisplay if onUIElement', function() {
            setupMiddleware({
                ui: {
                    onUIElement: true,
                },
            });
            expect(dispatch).toHaveBeenCalledTimes(1);
        });

        test('should call not setUIDisplay if paused', function() {
            setupMiddleware({
                playback: {
                    paused: true,
                },
            });
            expect(dispatch).toHaveBeenCalledTimes(1);
        });

        test('should call not setUIDisplay if offline', function() {
            setupMiddleware({
                onlineStatus: OFFLINE_STATUS,
            });
            expect(dispatch).toHaveBeenCalledTimes(1);
        });
    });

    describe('in response to actions', function() {
        // eslint-disable-next-line max-len
        test('on set miniplayer mode with truthy value, should dispatch hoveringOffUIElement, dispatch setUIDisplay with false and clear timer', function() {
            const updateOffUIElementAction = UIActions.hoveringOffUIElement();
            const setUIDisplayAction = UIActions.setUIDisplay(false);
            dispatchAction({
                type: UIActions.ACTION_SET_MINIPLAYER_MODE,
                value: true,
            });
            // eslint-disable-next-line max-len
            expect(dispatch.mock.calls[2][0]).toEqual(updateOffUIElementAction, 'hoveringOffUIElement should be called');
            // eslint-disable-next-line max-len
            expect(dispatch.mock.calls[3][0]).toEqual(setUIDisplayAction, 'setUIDisplay false should be called');
            expect(getState().window.clearTimeout).toHaveBeenCalled();
        });

        test('on pause, if not isMini, should dispatch setUIDisplay with true', function() {
            const action = UIActions.setUIDisplay(true);
            dispatchAction({
                type: PlaybackActions.ACTION_PAUSE,
            });
            expect(dispatch).toHaveBeenCalledTimes(3);
            expect(dispatch.mock.calls[2][0]).toEqual(action, 'dispatched action should be setUIDisplay true');
        });

        test('on pause, if isMini, should not dispatch setUIDisplay with true', function() {
            setupMiddleware({
                ui: {
                    isMini: true,
                },
            });
            dispatchAction({
                type: PlaybackActions.ACTION_PAUSE,
            });
            expect(dispatch).toHaveBeenCalledTimes(2);
        });

        // eslint-disable-next-line max-len
        test('on hide ui, should dispatch hoveringOffUIElement, dispatch setUIDisplay with false and clear timer when not in highlighter', function() {
            const updateOffUIElementAction = UIActions.hoveringOffUIElement();
            const setUIDisplayAction = UIActions.setUIDisplay(false);
            dispatchAction({
                type: UIActions.ACTION_HIDE_UI,
            });
            expect(dispatch.mock.calls[2][0]).toEqual(updateOffUIElementAction);
            expect(dispatch.mock.calls[3][0]).toEqual(setUIDisplayAction);
            expect(getState().window.clearTimeout).toHaveBeenCalled();
        });

        // eslint-disable-next-line max-len
        test('on hide ui, should not dispatch setUIDisplay, should clear timer when in highlighter', function() {
            setupMiddleware({
                env: {
                    playerType: PLAYER_HIGHLIGHTER,
                },
            });
            dispatchAction({
                type: UIActions.ACTION_HIDE_UI,
            });
            expect(dispatch).toHaveBeenCalledTimes(1);
            expect(getState().window.clearTimeout).toHaveBeenCalled();
        });

        test('on set casting state with connected, should dispatch setUIDisplay with true', function() {
            const action = UIActions.setUIDisplay(true);
            dispatchAction({
                type: ACTION_SET_CASTING_STATE,
                castingState: 'connected',
            });
            expect(dispatch.mock.calls[2][0]).toEqual(action);
        });

        test('on set casting state with connecting, should dispatch setUIDisplay with true', function() {
            const action = UIActions.setUIDisplay(true);
            dispatchAction({
                type: ACTION_SET_CASTING_STATE,
                castingState: 'connecting',
            });
            expect(dispatch.mock.calls[2][0]).toEqual(action);
        });

        test('on set casting state with other state, should reset timer with hover delay', function() {
            dispatchAction({
                type: ACTION_SET_CASTING_STATE,
                castingState: 'some other state',
            });
            expect(getState().window.clearTimeout).toHaveBeenCalled();
            expect(getState().window.setTimeout.mock.calls[1][1]).toBe(hoverControlsDelay);
        });

        function testShowAndHover(testAction) {
            // eslint-disable-next-line max-len
            test(`on ${testAction}, should call setUIDisplay with true, and reset timer with hover delay`, function() {
                const action = UIActions.setUIDisplay(true);
                dispatchAction({
                    type: testAction,
                });
                expect(dispatch.mock.calls[2][0]).toEqual(action, 'setUIDisplay true is called');
                expect(getState().window.clearTimeout).toHaveBeenCalled();
                expect(getState().window.setTimeout.mock.calls[1][1]).toBe(hoverControlsDelay);
            });
        }

        function testShowAndHoverMiniplayer(testAction) {
            // eslint-disable-next-line max-len
            test(`on ${testAction}, when isMini, should not call setUIDisplay, should reset timer with hover delay`, function() {
                setupMiddleware({
                    ui: {
                        isMini: true,
                    },
                });
                const action = UIActions.setUIDisplay(false);
                dispatchAction({
                    type: testAction,
                });
                expect(dispatch.mock.calls[2][0]).toEqual(action, 'setUIDisplay false is called');
                expect(getState().window.clearTimeout).toHaveBeenCalled();
                expect(getState().window.setTimeout.mock.calls[1][1]).toBe(hoverControlsDelay);
            });
        }

        function testShowAndHoverCollectionSideBar(testAction) {
            // eslint-disable-next-line max-len
            test(`on ${testAction}, when in collection sidebar view, should not call setUIDisplay, should reset timer with hover delay`, function() {
                setupMiddleware({
                    collection: {
                        currentView: SIDEBAR_VIEW,
                    },
                });
                const action = UIActions.setUIDisplay(false);
                dispatchAction({
                    type: testAction,
                });
                expect(dispatch.mock.calls[2][0]).toEqual(action, 'setUIDisplay false is called');
                expect(getState().window.clearTimeout).toHaveBeenCalled();
                expect(getState().window.setTimeout.mock.calls[1][1]).toBe(hoverControlsDelay);
            });
        }

        const showAndHoverActions = [
            PlaybackActions.ACTION_PLAYER_SEEKING,
            PlaybackActions.ACTION_PLAYING,
            ScreenActions.ACTION_SET_THEATRE_MODE,
            ScreenActions.ACTION_TOGGLE_THEATRE_MODE,
            UIActions.ACTION_SHOW_UI_WITH_AUTO_HIDE,
            ACTION_SET_CURRENT_QUALITY,
        ];

        showAndHoverActions.forEach(testShowAndHover);
        showAndHoverActions.forEach(testShowAndHoverMiniplayer);
        showAndHoverActions.forEach(testShowAndHoverCollectionSideBar);
    });
});
