import { ExtensionCoordinator } from 'extension-coordinator';
import React from 'react';
import assign from 'lodash/assign';
import merge from 'lodash/merge';
import { mount } from 'enzyme';
import { createExtensionParsed } from 'test-utils/fixtures/extensions';
import { ExtensionOverlay, EXT_OVERLAY_CLASS } from 'ui/components/extensions/extension-overlay';
import { EXTENSION_PERMISSION_STATE_NONE, EXTENSION_PERMISSION_STATE_GRANTED } from 'util/extensions';
import { createRandomStr } from 'test-utils/utils/create-random-string';

const DEFAULT_EXTENSION = createExtensionParsed()[0];
const DEFAULT_ARGS = Object.freeze({
    onDoubleClick() {},
    width: 300,
    height: 300,
    extension: DEFAULT_EXTENSION,
    game: 'LUL of Legends',
    language: 'zh',
    locale: 'zh-CN',
    onIdentityLinked() {},
    onModalRequested() {},
    trackingProperties: {},
    isLoggedIn: true,
});

function renderExtensionOverlay(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, {
        loginId: Math.floor(Math.random() * 10),
        trackingProperties: {
            locale: 'en',
            login: `user_${createRandomStr()}`,
            channel: `channel_${createRandomStr()}`,
            channelId: Math.floor(Math.random() * 10),
            deviceId: `device_${createRandomStr()}`,
            platform: 'site',
            playerType: 'popout',
        },
    }, overrides);
    const component = <ExtensionOverlay {...args} />;
    return mount(component);
}

describe('ui | components | extension-overlay', function() {
    test('returns a div with expected classname', function() {
        const component = renderExtensionOverlay();
        expect(component.childAt(0).type()).toBe('div');
        expect(component.childAt(0).hasClass(EXT_OVERLAY_CLASS)).toBeTruthy();
    });

    test('when width and height are set, they pass through to the child iframe', function() {
        const width = 800;
        const height = 600;
        const component = renderExtensionOverlay({
            width,
            height,
        });
        const hostElement = component.find(`.${EXT_OVERLAY_CLASS}`);
        const props = hostElement.props();
        expect(props.style.width).toBe(`${width}px`, 'iframe host width prop matches the component width prop');
        expect(props.style.height).toBe(`${height}px`, 'iframe host height prop matches the component height prop');
    });

    test('when an extension is set, the extension frame is created and destroyed properly', function() {
        const onIdentityLinkedSpy = jest.fn();
        const extension = createExtensionParsed()[0];
        const component = renderExtensionOverlay({
            extension,
            onIdentityLinked: onIdentityLinkedSpy,
        });

        const instance = component.instance();
        const frame = instance.extensionFrame;

        expect(frame).toBeTruthy();
        expect(frame.on).toHaveBeenCalledWith('identityLinked', instance._boundOnIdentityLinked);
        expect(ExtensionCoordinator.ExtensionService.postContext).toHaveBeenCalledWith({
            game: DEFAULT_ARGS.game,
        });

        instance._boundOnIdentityLinked();
        expect(onIdentityLinkedSpy).toHaveBeenCalled();

        component.unmount();

        expect(frame.destroy).toHaveBeenCalled();
        expect(frame.off).toHaveBeenCalledWith('identityLinked', instance._boundOnIdentityLinked);
    });

    test('when an extension is set, event listeners are added/removed correctly', function() {
        const extension = createExtensionParsed()[0];
        const component = renderExtensionOverlay({
            extension,
        });

        const instance = component.instance();
        const iframeHost = component.find(`.${EXT_OVERLAY_CLASS}`);
        const addEventListenerSpy = jest.fn();
        const removeEventListenerSpy = jest.fn();
        iframeHost.addEventListener = addEventListenerSpy;
        iframeHost.removeEventListener = removeEventListenerSpy;
        instance._boundIframeHostRefHandler(iframeHost);
        instance.componentDidMount();

        expect(addEventListenerSpy).toHaveBeenCalledTimes(1);
        expect(addEventListenerSpy.mock.calls[0][0]).toBe(
            'dblclick',
            'addEventListener was called with "dblclick" event type'
        );

        instance.componentWillUnmount();

        expect(removeEventListenerSpy).toHaveBeenCalledTimes(1);
        expect(removeEventListenerSpy.mock.calls[0][0]).toBe(
            'dblclick',
            'removeEventListener was called with "dblclick" event type'
        );
        expect(removeEventListenerSpy.mock.calls[0][1]).toBe(
            addEventListenerSpy.mock.calls[0][1],
            'removeEventListener was called with the same function instance as addEventListener'
        );
    });

    test('when lastUserIdentityLinkState differs from the linked state', function() {
        const extension = createExtensionParsed()[0];
        const component = renderExtensionOverlay({
            extension,
        });

        const instance = component.instance();
        const frame = instance.extensionFrame;

        component.setProps({
            extension: merge({}, extension, {
                lastUserIdentityLinkState: true,
                token: {
                    permissionsState: EXTENSION_PERMISSION_STATE_NONE,
                },
            }),
        });

        expect(frame.linkIdentity).toHaveBeenCalled();

        component.setProps({
            extension: merge({}, extension, {
                lastUserIdentityLinkState: false,
                token: {
                    permissionsState: EXTENSION_PERMISSION_STATE_GRANTED,
                },
            }),
        });

        expect(frame.unlinkIdentity).toHaveBeenCalled();
    });

    test('when language is specified', function() {
        const extension = createExtensionParsed()[0];
        const component = renderExtensionOverlay({
            extension,
        });

        const instance = component.instance();
        const frame = instance.extensionFrame;

        expect(frame.language).toBe(
            DEFAULT_ARGS.language,
            'language prop is passed through to ExtensionFrame as language'
        );
    });

    test('when locale is specified', function() {
        const extension = createExtensionParsed()[0];
        const component = renderExtensionOverlay({
            extension,
        });

        const instance = component.instance();
        const frame = instance.extensionFrame;

        expect(frame.locale).toBe(
            DEFAULT_ARGS.locale,
            'locale prop is passed through to ExtensionFrame as locale'
        );
    });
});
