import React from 'react';
import assign from 'lodash/assign';
import { shallow } from 'enzyme';
import { ClosedCaptionsContainer, mapStateToProps } from 'ui/containers/closed-captions/closed-captions';
import { ClosedCaptions as ClosedCaptionsComponent } from 'ui/components/closed-captions/closed-captions';

const BASE_STYLES = Object.freeze({
    fontSize: 20,
    font: 'prop-sans-serif',
    fontOpacity: 'solid',
    alignment: 'left',
    edge: 'none',
    verticalPosition: 'bottom',
    backgroundColorName: 'black',
    backgroundOpacity: 'solid',
    windowColorName: 'transparent',
    windowOpacity: 'solid',
    fontUnderline: 'none',
    fontBold: 'none',
    fontItalic: 'none',
});

const DEFAULT_ARGS = Object.freeze({
    enabled: false,
    captionsData: null,
    captionsStyle: BASE_STYLES,
});

const CAPTIONS_DATA = Object.freeze({
    format: 'eia608',
    mode: 'pop-on',
    rollUp: 0,
    data: [
        /* eslint-disable object-property-newline */
        { row: 0, col: 0, char: 'K', style: 'white' },
        { row: 0, col: 1, char: 'i', style: 'white' },
        { row: 0, col: 2, char: 't', style: 'white' },
        { row: 0, col: 3, char: ' ', style: 'white' },
        { row: 0, col: 4, char: 'K', style: 'white' },
        { row: 0, col: 5, char: 'a', style: 'white' },
        { row: 0, col: 6, char: 't', style: 'white' },
        { row: 1, col: 0, char: '', style: 'white' },
        { row: 2, col: 0, char: 'D', style: 'white' },
        { row: 2, col: 1, char: 'o', style: 'white' },
        { row: 2, col: 2, char: 'g', style: 'white' },
    ], /* eslint-enable object-property-newline */
});

function renderClosedCaptionsContainer(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <ClosedCaptionsContainer {...args} />;
    return shallow(component);
}

describe('ui | containers | closed-captions | closed-captions', function() {
    test('renders null if not enabled', function() {
        const component = renderClosedCaptionsContainer();
        expect(component.type()).toBe(null, 'renders null if not enabled');
    });

    test('renders null if captions data is empty', function() {
        const component = renderClosedCaptionsContainer({
            enabled: true,
            captionsData: {
                data: [],
            },
        });
        expect(component.type()).toBe(null, 'renders null if no data present');
    });

    test('renders captions component with correct props set', function() {
        const component = renderClosedCaptionsContainer({
            enabled: true,
            captionsData: CAPTIONS_DATA,
        });

        expect(component.type()).toBe(ClosedCaptionsComponent, 'renders closed captions component');

        expect(component.props().captionsText).toEqual(['Kit Kat', '', 'Dog'], 'formats captions text correctly');

        expect(component.props().captionsStyle).toEqual(BASE_STYLES, 'styles are passed in');
    });

    test('parses captions data if given captions string', function() {
        const component = renderClosedCaptionsContainer({
            enabled: true,
            captionsData: {
                text: 'Hello\nWorld\nFooBar',
            },
        });

        expect(component.type()).toBe(ClosedCaptionsComponent, 'renders closed captions component');

        expect(component.props().captionsText).toEqual(['Hello', 'World', 'FooBar'], 'formats captions text correctly');
    });

    describe('mapStateToProps', () => {
        test('enabled is false if in mini mode', () => {
            const props = mapStateToProps({
                ui: {
                    isMini: true,
                },
                captions: {
                    enabled: true,
                    data: {},
                    captionsStyle: {},
                },
            });
            expect(props.enabled).toBe(false);
        });
    });
});
