import React from 'react';
import { shallow } from 'enzyme';
import assign from 'lodash/assign';
import {
    ALERT_MESSAGE,
    MutedSegmentsAlertComponent,
    mapStateToProps,
} from 'ui/containers/muted-segments-alert';
import { CONTENT_SCREEN, ERROR_SCREEN } from 'actions/screen';
import { mockTranslateFunc } from 'test-utils/utils/translate-hoc-helpers';

const DEFAULT_ARGS = Object.freeze({
    t: mockTranslateFunc,
    onHoverElement: () => {},
    showUIWithAutoHide: () => {},
    mutedSegmentsAvailable: true,
    uiShowing: true,
    isShowingContentScreen: true,
});

function renderMutedSegmentsAlert(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    return shallow(<MutedSegmentsAlertComponent {...args} />);
}

describe('ui | containers | muted-segments-alert', () => {
    describe('correctly maps state to props', () => {
        const state = {
            timelineMetadata: {
                mutedSegments: [],
            },
            ui: {
                showUI: false,
            },
            screen: [CONTENT_SCREEN],
            playback: {
                paused: false,
            },
        };

        test('mutedSegmentsAvailable', () => {
            state.timelineMetadata.mutedSegments = [];
            expect(mapStateToProps(state).mutedSegmentsAvailable).toBe(false);

            state.timelineMetadata.mutedSegments = [{}, {}];
            expect(mapStateToProps(state).mutedSegmentsAvailable).toBe(true);
        });

        test('uiShowing', () => {
            state.ui.showUI = true;
            expect(mapStateToProps(state).uiShowing).toBe(true);
        });

        test('mutedSegmentsAvailable', () => {
            state.screen = [CONTENT_SCREEN];

            expect(mapStateToProps(state).isShowingContentScreen).toBe(true);

            state.screen = [ERROR_SCREEN];
            expect(mapStateToProps(state).isShowingContentScreen).toBe(false);
        });
    });

    test('renders null if no muted segments are available', () => {
        const component = renderMutedSegmentsAlert({
            mutedSegmentsAvailable: false,
        });

        expect(component.type()).toBe(null);
    });

    test('renders null if content screen is not showing', () => {
        const component = renderMutedSegmentsAlert({
            isShowingContentScreen: false,
        });

        expect(component.type()).toBe(null);
    });

    test('renders with expected css class', () => {
        const component = renderMutedSegmentsAlert();

        expect(component.hasClass('player-alert')).toBe(true);
        expect(component.hasClass('muted-alert-on-ui')).toBe(true);
    });

    test('renders with correct onMouse handlers', () => {
        const component = renderMutedSegmentsAlert();

        expect(component.prop('onMouseEnter')).toBe(DEFAULT_ARGS.onHoverElement);
        expect(component.prop('onMouseLeave')).toBe(DEFAULT_ARGS.showUIWithAutoHide);
    });

    test('renders alert message', () => {
        const component = renderMutedSegmentsAlert();
        const msgComponent = component.childAt(0);

        expect(msgComponent.type()).toBe('p');
        expect(msgComponent.text()).toBe(DEFAULT_ARGS.t(ALERT_MESSAGE));
    });

    test('renders close button', () => {
        const component = renderMutedSegmentsAlert();
        const buttonComponent = component.childAt(1);
        const svgComponent = buttonComponent.childAt(0);

        expect(buttonComponent.hasClass('player-button')).toBe(true);
        expect(buttonComponent.hasClass('player-button--noscale')).toBe(true);
        expect(buttonComponent.hasClass('player-button--close')).toBe(true);

        expect(svgComponent.type()).toBe('svg');
    });
});
