import React from 'react';
import assign from 'lodash/assign';
import { shallow } from 'enzyme';
import { ACTION_CONTENT_IS_SHOWING, ACTION_CLEAR_QUALITY_RESTRICTED_ERROR  } from 'actions/playback';
import { ACTION_POP_SCREEN, ACTION_PUSH_SCREEN , QUALITY_RESTRICTED_SCREEN } from 'actions/screen';
import { QualityRestrictedContainer,
         mapStateToProps,
         mapDispatchToProps } from 'ui/containers/overlays/quality-restricted-overlay';
import {
    QualityRestrictedOverlay as QualityRestrictedComponent,
} from 'ui/components/overlays/quality-restricted-overlay';

const DEFAULT_ARGS = {
    channelName: '',
    contentIsShowing() {},
    popScreen() {},
    dispatchQualityRestrictedScreen() {},
    clearQualityRestrictedError() {},
};

function renderQualityRestrictedOverlay(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <QualityRestrictedContainer {...args} />;
    return shallow(component);
}

describe('ui | containers | overlays | quality restricted', () => {
    test('renders QualityRestrictedContainer when qualtiy restricted', function() {
        const component = renderQualityRestrictedOverlay({
            shouldRenderQualityOverlay: true,
        });
        expect(component.type()).toBe(QualityRestrictedComponent);
    });

    test('renders null when showQualityRestrictedOverlay is false', function() {
        const component = renderQualityRestrictedOverlay();
        expect(component.type()).toBe(null);
    });

    test('onClose', () => {
        const clearQualityRestrictedErrorSpy = jest.fn();
        const popScreenSpy = jest.fn();
        const component = renderQualityRestrictedOverlay({
            shouldRenderQualityOverlay: true,
            clearQualityRestrictedError: clearQualityRestrictedErrorSpy,
            popScreen: popScreenSpy,
        });
        component.instance().onClose();
        expect(clearQualityRestrictedErrorSpy).toHaveBeenCalledTimes(1);
        expect(popScreenSpy).toHaveBeenCalledTimes(1);
    });

    test('componentDidMount', () => {
        const contentIsShowingSpy = jest.fn();
        const component = renderQualityRestrictedOverlay({
            shouldRenderQualityOverlay: true,
            contentIsShowing: contentIsShowingSpy,
        });
        contentIsShowingSpy.mockReset();
        component.instance().componentDidMount();
        expect(contentIsShowingSpy).toHaveBeenCalledTimes(1);
    });

    describe('componentWillReceiveProps', () => {
        let component;
        let dispatchQualityRestrictedScreenSpy;

        beforeEach(() => {
            dispatchQualityRestrictedScreenSpy = jest.fn();
            component = renderQualityRestrictedOverlay({
                shouldRenderQualityOverlay: true,
                dispatchQualityRestrictedScreen: dispatchQualityRestrictedScreenSpy,
            });
            dispatchQualityRestrictedScreenSpy.mockReset();
        });

        test('dispatches restricted quality screen when should shouldRenderQualityOverlay is true', () => {
            component.instance().componentWillReceiveProps(assign({}, DEFAULT_ARGS, {
                shouldRenderQualityOverlay: true,
                dispatchQualityRestrictedScreen: dispatchQualityRestrictedScreenSpy,
            }));
            expect(dispatchQualityRestrictedScreenSpy).toHaveBeenCalledTimes(1);
        });

        test('does not dispatch restricted quality screen when should shouldRenderQualityOverlay is false', () => {
            component.instance().componentWillReceiveProps(assign({}, DEFAULT_ARGS, {
                shouldRenderQualityOverlay: false,
                dispatchQualityRestrictedScreen: dispatchQualityRestrictedScreenSpy,
            }));
            expect(dispatchQualityRestrictedScreenSpy).toHaveBeenCalledTimes(0);
        });
    });

    describe('mapDispatchToProps', () => {
        test('contentIsShowing', function() {
            const dispatchSpy = jest.fn();
            mapDispatchToProps(dispatchSpy).contentIsShowing();
            expect(dispatchSpy).toHaveBeenCalledTimes(1);
            expect(dispatchSpy).toHaveBeenCalledWith({
                type: ACTION_CONTENT_IS_SHOWING,
            });
        });

        test('popScreen', function() {
            const dispatchSpy = jest.fn();
            mapDispatchToProps(dispatchSpy).popScreen();
            expect(dispatchSpy).toHaveBeenCalledTimes(1);
            expect(dispatchSpy).toHaveBeenCalledWith({
                type: ACTION_POP_SCREEN,
            });
        });

        test('dispatchQualityRestrictedScreen', function() {
            const dispatchSpy = jest.fn();
            mapDispatchToProps(dispatchSpy).dispatchQualityRestrictedScreen();
            expect(dispatchSpy).toHaveBeenCalledTimes(1);
            expect(dispatchSpy).toHaveBeenCalledWith({
                type: ACTION_PUSH_SCREEN,
                screen: QUALITY_RESTRICTED_SCREEN,
            });
        });

        test('clearQualityRestrictedError', function() {
            const dispatchSpy = jest.fn();
            mapDispatchToProps(dispatchSpy).clearQualityRestrictedError();
            expect(dispatchSpy).toHaveBeenCalledTimes(1);
            expect(dispatchSpy).toHaveBeenCalledWith({
                type: ACTION_CLEAR_QUALITY_RESTRICTED_ERROR,
            });
        });
    });

    describe('mapStateToProps', () => {
        let state;
        beforeEach(() => {
            state = {
                playback: {
                    restrictedQualityError: true,
                },
                streamMetadata: {
                    channel: {
                        name: 'monsterTest',
                    },
                },
            };
        });
        test('sets shouldRenderQualityOverlay correctly', function() {
            let result = mapStateToProps(state);
            expect(result.shouldRenderQualityOverlay).toBe(true);

            const falseState = assign(state, {
                playback: {
                    restrictedQualityError: false,
                },
            });

            result = mapStateToProps(falseState);
            expect(result.shouldRenderQualityOverlay).toBe(false);
        });

        test('sets channelName correctly', function() {
            let result = mapStateToProps(state);
            expect(result.channelName).toBe('monsterTest');

            const twitchmediaChannelState = assign(state, {
                streamMetadata: {
                    channel: {
                        name: 'twitchmedia2',
                    },
                },
            });
            result = mapStateToProps(twitchmediaChannelState);
            expect(result.channelName).toBe('twitchmedia2');
        });
    });
});
