import React from 'react';
import assign from 'lodash/assign';
import { shallow } from 'enzyme';
import renderer from 'react-test-renderer';
import { ADVERTISEMENT_SCREEN } from 'actions/screen';
import { PLAYER_FRONTPAGE } from 'util/player-type';
import { ToggleFullscreenContainer,
         mapDispatchToProps,
         mapStateToProps } from 'ui/containers/overlays/toggle-fullscreen-overlay';
import { ToggleFullscreen } from 'ui/components/overlays/toggle-fullscreen-overlay';

const DEFAULT_ARGS = {
    trackEvent() {},
    isFullscreen: false,
    shouldRenderFullscreenOverlay: true,
    toggleFullScreen() {},
    isAd: false,
    showUI: false,
};

function renderToggleFullscreenOverlay(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <ToggleFullscreenContainer {...args} />;
    return shallow(component);
}

describe('ui | containers | overlays | toggle fullscreen', () => {
    test('snapshot is correctly matching', () => {
        const tree = renderer.
            create(<ToggleFullscreenContainer {...DEFAULT_ARGS} />).
            toJSON();
        expect(tree).toMatchSnapshot();
    });

    test('renders ToggleFullscreenOverlay with correct state', () => {
        const component = renderToggleFullscreenOverlay({
            shouldRenderFullscreenOverlay: true,
            isAd: false,
            showUI: false,
        });
        expect(component.type()).toBe(ToggleFullscreen);
    });

    test('does not render ToggleFullscreenOverlay when advertisment is showing', () => {
        const component = renderToggleFullscreenOverlay({
            shouldRenderFullscreenOverlay: true,
            isAd: true,
            showUI: false,
        });
        expect(component.type()).toBe(null);
    });

    test('does not render ToggleFullscreenOverlay when ui is showing', () => {
        const component = renderToggleFullscreenOverlay({
            shouldRenderFullscreenOverlay: true,
            isAd: false,
            showUI: true,
        });
        expect(component.type()).toBe(null);
    });

    test('double clicking will call toggle fullscreen correctly', () => {
        const toggleFullScreenSpy = jest.fn();
        const component = renderToggleFullscreenOverlay({
            shouldRenderFullscreenOverlay: true,
            isAd: false,
            showUI: false,
            toggleFullScreen: toggleFullScreenSpy,
        });
        const instance = component.instance();
        instance.handleDoubleClick();
        expect(toggleFullScreenSpy).toHaveBeenCalledTimes(1);
    });

    test('double clicking will call track event correctly', () => {
        const trackEventSpy = jest.fn();
        const component = renderToggleFullscreenOverlay({
            shouldRenderFullscreenOverlay: true,
            isAd: false,
            showUI: false,
            trackEvent: trackEventSpy,
        });
        const instance = component.instance();
        instance.handleDoubleClick();
        expect(trackEventSpy).toHaveBeenCalledTimes(1);
    });

    describe('mapDispatchToProps', () => {
        test('toggleFullScreen can be dispatched', () => {
            const dispatchSpy = jest.fn();
            mapDispatchToProps(dispatchSpy).toggleFullScreen();
            expect(dispatchSpy).toHaveBeenCalledTimes(1);
        });
    });

    describe('mapStateToProps', () => {
        const state = {
            analyticsTracker: {
                trackEvent() {},
            },
            screenMode: {
                isFullScreen: false,
            },
            screen: [ADVERTISEMENT_SCREEN],
            ui: {
                showUI: false,
            },
            env: {
                playerType: PLAYER_FRONTPAGE,
            },
        };
        test('sets shouldRenderQualityOverlay correctly', function() {
            const props = mapStateToProps(state);
            expect(props.shouldRenderFullscreenOverlay).toBe(false);
        });

        test('sets trackEvent correctly', () => {
            const props = mapStateToProps(state);
            expect(props.trackEvent).toEqual(state.analyticsTracker.trackEvent);
        });

        test('sets isFullScreen correctly', () => {
            const props = mapStateToProps(state);
            expect(props.isFullScreen).toBe(true);
        });

        test('sets isAd correctly', () => {
            const props = mapStateToProps(state);
            expect(props.isAd).toBe(true);
        });

        test('sets showUI correctly', () => {
            const props = mapStateToProps(state);
            expect(props.showUI).toBe(false);
        });
    });
});
