import {
    mapStateToProps,
    VodPostplayRecommendationsContainer,
}  from 'ui/containers/recommendations/post-vod-recommendations';
import { PostVODRecommendations } from 'ui/components/recommendations/post-vod-recommendations';
import { NORMALIZED_V3_RECOMMENDED_VIDEOS } from 'test-utils/fixtures/recommendations';
import { VOD_RECOMMENDATION_SCREEN, CONTENT_SCREEN } from 'actions/screen';
import { POST_VOD_RECOMMENDATIONS_TYPE, FETCHED, UNFETCHED } from 'actions/recommendations';
import * as PlayerType from 'util/player-type';
import { TRANSITION_TYPE_RECOMMENDATIONS } from 'state/playback';
import { CONTENT_MODE_VOD } from 'stream/twitch-vod';
import { setupShallowTest } from 'test-utils/utils/helpers';

describe('ui | containers | VodPostplayRecommendationsContainer', () => {
    const setup = setupShallowTest(VodPostplayRecommendationsContainer, () => ({
        hasVideoEnded: false,
        recommendedVideos: [],
        windowObj: {},
        i18n: {},
        playerDimensions: {
            height: 100,
            width: 100,
        },
        trackEvent: jest.fn(),
        isMiniPlayer: false,
        shouldShowRecommendations: false,
        shouldFetchRecommendations: false,
        screen: [CONTENT_SCREEN],
        fetchRecommendedVODs: jest.fn(),
        pushVODRecScreen: jest.fn(),
        selectRecommendedVideo: jest.fn(),
        popScreen: jest.fn(),
        clearRecommendedVODs: jest.fn(),
    }));

    test('renders PostVODRecommendations if the first screen is VOD_RECOMMENDATION_SCREEN', function() {
        const { wrapper, props } = setup({
            screen: [VOD_RECOMMENDATION_SCREEN],
            recommendedVideos: NORMALIZED_V3_RECOMMENDED_VIDEOS,
        });

        expect(wrapper.find(PostVODRecommendations).length).toEqual(1);
        expect(wrapper.find(PostVODRecommendations).prop('videos')).toEqual(props.recommendedVideos);
    });

    test('renders nothing if first screen is not VOD_RECOMMENDATION_SCREEN', function() {
        const { wrapper } = setup();

        expect(wrapper.find(PostVODRecommendations).exists()).toEqual(false);
    });

    test('renders nothing if there are no recommended videos', function() {
        const { wrapper } = setup({
            hasVideoEnded: true,
            shouldShowRecommendations: true,
            recommendedVideos: [],
        });

        expect(wrapper.find(PostVODRecommendations).exists()).toEqual(false);
    });

    test('when there are new videos with no previews, we call _preloadVideoThumbnails' , () => {
        const { wrapper } = setup({
            recommendedVideos: [],
        });

        expect(wrapper.state('isImagePreloadingComplete')).toEqual(false);

        wrapper.setProps({
            recommendedVideos: NORMALIZED_V3_RECOMMENDED_VIDEOS,
        });

        expect(wrapper.state('isImagePreloadingComplete')).toEqual(true);
    });

    test('should push screen, when recommendations just loaded and video has already ended', () => {
        const { wrapper, props } = setup({
            recommendedVideos: [],
            hasVideoEnded: true,
        });

        expect(props.pushVODRecScreen).not.toHaveBeenCalled();

        wrapper.setProps({
            recommendedVideos: NORMALIZED_V3_RECOMMENDED_VIDEOS,
            shouldShowRecommendations: true,
        });

        expect(props.pushVODRecScreen).toHaveBeenCalled();
    });

    test('should push screen, when video just ended and recommendations already exist', () => {
        const { wrapper, props } = setup({
            recommendedVideos: NORMALIZED_V3_RECOMMENDED_VIDEOS,
            hasVideoEnded: false,
        });

        expect(props.pushVODRecScreen).not.toHaveBeenCalled();

        wrapper.setProps({
            hasVideoEnded: true,
            shouldShowRecommendations: true,
        });

        expect(props.pushVODRecScreen).toHaveBeenCalled();
    });

    test('calls fetchRecommendedVODs when shouldFetchRecommendations is true', () => {
        const { wrapper, props } = setup({
            recommendedVideos: [],
            shouldFetchRecommendations: false,
        });

        expect(props.fetchRecommendedVODs).not.toHaveBeenCalled();

        wrapper.setProps({
            shouldFetchRecommendations: true,
        });

        expect(props.fetchRecommendedVODs).toHaveBeenCalled();
    });

    // eslint-disable-next-line max-len
    test('pops the recommendation screen if nextProps.shouldShowRecommendations becomes false, ie. when the player seeks when on recommendation screen', () => {
        const { wrapper, props } = setup({
            shouldShowRecommendations: true,
            screen: [CONTENT_SCREEN],
        });

        wrapper.setProps({
            shouldShowRecommendations: false,
            screen: [VOD_RECOMMENDATION_SCREEN],
        });

        expect(props.popScreen).toHaveBeenCalled();
    });

    test('does not show recommendations again after exiting recommendations screen', () => {
        const { wrapper, props } = setup({
            screen: [VOD_RECOMMENDATION_SCREEN],
        });

        wrapper.find(PostVODRecommendations).prop('onExit')();

        expect(props.popScreen).toHaveBeenCalled();
        wrapper.setProps({ screen: [CONTENT_SCREEN] });

        expect(wrapper.find(PostVODRecommendations)).toHaveLength(0);
    });

    test('onSelectVOD prop on post vod recommendations selects the vod, pops the screen', () => {
        const { wrapper, props } = setup({
            screen: [VOD_RECOMMENDATION_SCREEN],
            recommendedVideos: NORMALIZED_V3_RECOMMENDED_VIDEOS,
        });

        wrapper.find(PostVODRecommendations).prop('onSelectVOD')(props.recommendedVideos[0]);

        expect(props.selectRecommendedVideo).toHaveBeenCalledWith(props.recommendedVideos[0].id);
        expect(props.popScreen).toHaveBeenCalled();

        wrapper.setProps({ screen: [CONTENT_SCREEN] });

        expect(wrapper.find(PostVODRecommendations)).toHaveLength(0);
    });

    describe('#mapStateToProps', () => {
        const state = {
            lang: {},
            screen: [VOD_RECOMMENDATION_SCREEN],
            recommendations: {
                videos: NORMALIZED_V3_RECOMMENDED_VIDEOS,
                type: POST_VOD_RECOMMENDATIONS_TYPE,
                status: FETCHED,
            },
            window: {},
            playerDimensions: {
                height: 100,
                width: 100,
            },
            analyticsTracker: {
                trackEvent: jest.fn(),
            },
            ui: {
                isMini: false,
            },
            playback: {
                currentTime: 0,
                duration: 0,
                ended: true,
                transitionScheme: TRANSITION_TYPE_RECOMMENDATIONS,
            },
            env: {
                playerType: PlayerType.PLAYER_POPOUT,
            },
            stream: {
                contentType: CONTENT_MODE_VOD,
            },
        };

        describe('shouldShowRecommendations property', () => {
            test(`should be true all of the conditions are true:
                stream content mod is vod,
                playback has ended,
                ui is not mini,
                does not include invalid playertypes
                recommendation status is fetched
                playback transition scheme is TRANSITION_TYPE_RECOMMENDATIONS
                recommendations type is POST_VOD_RECOMMENDATIONS_TYPE
            `, () => {
                expect(mapStateToProps(state).shouldShowRecommendations).toEqual(true);
            });
        });

        describe('shouldFetchRecommendations property', () => {
            test(`should be true if all of the conditions are true:
                playback transition scheme is TRANSITION_TYPE_RECOMMENDATIONS
                stream content mod is vod,
                we are 10 seconds away from end
                recommendation status is unfetched
            `, () => {
                const testState = Object.assign({}, state, { recommendations: { status: UNFETCHED } });
                expect(mapStateToProps(testState).shouldFetchRecommendations).toEqual(true);
            });
        });
    });
});
