import React from 'react';
import assign from 'lodash/assign';
import { shallow } from 'enzyme';
import { SeekbarComponent, classNames, mapDispatchToProps } from 'ui/containers/seekbar';
import { SeekbarTimeDisplay } from 'ui/components/seekbar/seekbar-time-display';
import { SeekbarBuffer } from 'ui/components/seekbar/seekbar-buffer';
import { SeekbarMarkers } from 'ui/components/seekbar/seekbar-markers';
import { MutedSegments } from 'ui/components/seekbar/muted-segments';
import { ThumbnailPreviews } from 'ui/components/seekbar/thumbnail-previews';
import { Slider } from 'ui/components/slider';
import { seek } from 'actions/video-api';
import { setLoading } from 'actions/playback';

const DEFAULT_ARGS = Object.freeze({
    buffer: {
        start: 30,
        end: 40,
    },
    currentTime: 0,
    duration: 100,
    getCurrentTime: () => {},
    isLoading: false,
    isPaused: false,
    isSeekableStream: true,
    isClipsPlayerType: false,
    mutedSegments: [],
    seek() {},
    trackEvent() {},
    seekbarMarkers: [],
    thumbnailPreviews: {
        count: 0,
    },
    windowObj: {},
});

function renderSeekbar(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <SeekbarComponent {...args} />;
    return shallow(component);
}

describe('ui | containers | seekbar', () => {
    test('returns a div', function() {
        const component = renderSeekbar();
        expect(component.type()).toBe('div');
    });

    test('renders nothing if not a seekable stream', function() {
        const component = renderSeekbar({
            isSeekableStream: false,
        });

        expect(component.type()).toBe(null);
    });

    test('contains a time display component', function() {
        const component = renderSeekbar();
        expect(component.containsMatchingElement(
            <SeekbarTimeDisplay />
        )).toBe(true);
    });

    test('contains a time display component', function() {
        const component = renderSeekbar();
        expect(component.containsMatchingElement(
            <SeekbarTimeDisplay
                currentTime={DEFAULT_ARGS.currentTime}
                duration={DEFAULT_ARGS.duration}
            />
        )).toBe(true);
    });

    describe('Slider component', function() {
        test('contains a slider component with expected args', function() {
            const component = renderSeekbar();

            expect(component.containsMatchingElement(
                <Slider
                    max={DEFAULT_ARGS.duration}
                    min={0}
                    value={DEFAULT_ARGS.currentTime}
                    classNames={classNames}
                    getOptimizedValue={DEFAULT_ARGS.getCurrentTime}
                    valueOptimizationEnabled={true}
                    windowObj={DEFAULT_ARGS.windowObj}
                />
            )).toBe(true);
        });

        test('contains a slider component with correct components after slider children', function() {
            const component = renderSeekbar();
            const afterSliderChildren = component.childAt(1).prop('afterSliderChildren');

            const mutedsegmentsComponent = afterSliderChildren[0];
            const seekbarMarkersComponent = afterSliderChildren[2];

            expect(mutedsegmentsComponent.type).toEqual(MutedSegments);
            expect(mutedsegmentsComponent.props).toEqual({
                duration: DEFAULT_ARGS.duration,
                mutedSegments: DEFAULT_ARGS.mutedSegments,
            });

            expect(seekbarMarkersComponent.type).toEqual(SeekbarMarkers);
            expect(seekbarMarkersComponent.props.currentTime).toBe(DEFAULT_ARGS.currentTime);
            expect(seekbarMarkersComponent.props.duration).toBe(DEFAULT_ARGS.duration);
            expect(seekbarMarkersComponent.props.markers).toBe(DEFAULT_ARGS.seekbarMarkers);
        });

        // eslint-disable-next-line max-len
        test('if previews are available, display thumbnail previews on mouse over', function() {
            const thumbnailPreviews = { count: 1 };
            const component = renderSeekbar({
                thumbnailPreviews,
            });

            component.
                find(Slider).
                simulate('mousemove', {
                    clientX: 0,
                });

            const afterSliderChildren = component.childAt(1).prop('afterSliderChildren');
            const thumbnailPreviewsComponent = afterSliderChildren[1];

            expect(thumbnailPreviewsComponent.type).toEqual(ThumbnailPreviews);
            expect(thumbnailPreviewsComponent.props).toEqual({
                duration: DEFAULT_ARGS.duration,
                previews: thumbnailPreviews,
                mouseMoveClientX: 0,
                seekbarLeftOffset: 0,
                hideThumbnailPreview: false,
                seekbarWidth: 0,
            });
        });

        test('has an onClick prop that invokes seekToPosition', function() {
            const clickValue = 5;
            const component = renderSeekbar();
            const instance = component.instance();
            jest.spyOn(instance, 'seekToPosition');
            const sliderComponent = component.find(Slider);

            expect(instance.seekToPosition).toHaveBeenCalledTimes(0);
            component.state().isDragging = true;
            sliderComponent.props().onClick(clickValue);
            expect(instance.seekToPosition).toHaveBeenCalledWith(clickValue);
        });

        test('has a drag start handler that sets `isDragging` to true', function() {
            const component = renderSeekbar();
            const sliderComponent = component.find(Slider);

            expect(component.state().isDragging).toBeFalsy();
            sliderComponent.props().dragHandlers.onStart();
            expect(component.state().isDragging).toBe(true);
        });

        test('has a drag stop handler that sets `isDragging` to false', function() {
            const dragStopValue = 5;
            const component = renderSeekbar();
            const sliderComponent = component.find(Slider);

            component.setState({
                isDragging: true,
            });
            sliderComponent.props().dragHandlers.onStop(dragStopValue);
            expect(component.state().isDragging).toBeFalsy();
        });

        // eslint-disable-next-line max-len
        test('seekbar buffer included behind slider children when not dragging, and not included when dragging', function() {
            const component = renderSeekbar();
            const seekbarBufferComponent = component.childAt(1).prop('behindSliderChildren');

            expect(seekbarBufferComponent.type).toEqual(SeekbarBuffer);
            expect(seekbarBufferComponent.props).toEqual({
                min: 0,
                max: DEFAULT_ARGS.duration,
                start: DEFAULT_ARGS.buffer.start,
                end: DEFAULT_ARGS.buffer.end,
            });

            component.setState({
                isDragging: true,
            });

            expect(component.childAt(1).prop('behindSliderChildren')).toBe(null);
        });

        test('has a drag stop handler that invokes seekToPosition', function() {
            const dragStopValue = 5;
            const component = renderSeekbar();
            const instance = component.instance();
            jest.spyOn(instance, 'seekToPosition');
            const sliderComponent = component.find(Slider);

            expect(instance.seekToPosition).toHaveBeenCalledTimes(0);
            sliderComponent.props().dragHandlers.onStop(dragStopValue);
            expect(instance.seekToPosition).toHaveBeenCalledWith(dragStopValue);
        });
    });

    test('seekToPosition calls seek prop', function() {
        const seekedValue = 5;
        const seekSpy = jest.fn();
        const component = renderSeekbar({
            seek: seekSpy,
        });
        const instance = component.instance();

        expect(seekSpy).toHaveBeenCalledTimes(0);
        instance.seekToPosition(seekedValue);
        expect(seekSpy).toHaveBeenCalledWith(seekedValue);
    });

    describe('mapDispatchToProps', function() {
        test('seek dispatches seek middleware action and setLoading', function() {
            const dispatchSpy = jest.fn();
            const seekTime = 100;
            mapDispatchToProps(dispatchSpy).seek(seekTime);

            expect(dispatchSpy).toHaveBeenCalledTimes(2);
            expect(dispatchSpy.mock.calls[0][0]).toEqual(seek(seekTime));
            expect(dispatchSpy.mock.calls[1][0]).toEqual(setLoading(true));
        });
    });
});
