import React from 'react';
import assign from 'lodash/assign';
import { shallow } from 'enzyme';
import { TopBarContainer, mapStateToProps } from 'ui/containers/top-bar';
import { PLAYER_AMAZON_VSE, PLAYER_SITE } from 'util/player-type';

const DEFAULT_ARGS = Object.freeze({
    children: [<div id="testDiv" key="first" />,
        <div id="testDiv2" key="second" />],
    showTopBar: false,
});

function renderTopBar(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    return shallow(<TopBarContainer {...args} />);
}

describe('ui | containers | top-bar', function() {
    test('renders null if showTopBar is false', function() {
        const component = renderTopBar();
        expect(component.type()).toBe(null);
    });

    test('mapStateToProps', function() {
        const playerSite = {
            env: {
                playerType: PLAYER_SITE,
            },
            ui: {
                isMini: false,
            },
        };

        const miniPlayer = {
            env: {
                playerType: PLAYER_SITE,
            },
            ui: {
                isMini: true,
            },
        };

        const playerAmazonVSE = {
            env: {
                playerType: PLAYER_AMAZON_VSE,
            },
            ui: {
                isMini: false,
            },
        };

        const playerSiteProps = mapStateToProps(playerSite);
        const playerAmazonVSEProps = mapStateToProps(playerAmazonVSE);
        const miniPlayerProps = mapStateToProps(miniPlayer);

        expect(miniPlayerProps).toEqual({ showTopBar: false });
        expect(playerSiteProps).toEqual({ showTopBar: true });
        expect(playerAmazonVSEProps).toEqual({ showTopBar: false });
    });

    test('renders a div with proper classNames', function() {
        const component = renderTopBar({ showTopBar: true });
        expect(component.type()).toBe('div');
        expect(component.hasClass('pl-controls-top')).toBe(true, 'has pl-controls-top class');
        expect(component.hasClass('js-controls-top')).toBe(true, 'has js-controls-top class');
    });

    test('should have all passed in children', function() {
        const component = renderTopBar({ showTopBar: true });
        expect(component.children().length).toBe(DEFAULT_ARGS.children.length);
        const firstChild = component.childAt(0);
        const secondChild = component.childAt(1);

        expect(firstChild.matchesElement(DEFAULT_ARGS.children[0])).toBe(true, 'should have the correct first child');
        expect(secondChild.matchesElement(DEFAULT_ARGS.children[1])).toBe(true, 'should have the correct second child');
    });
});
