import React from 'react';
import assign from 'lodash/assign';
import { shallow } from 'enzyme';
import { mapStateToProps, StreamInfoComponent } from 'ui/containers/top-bar/stream-info';
import { LinkedImage } from 'ui/components/common/linked-image';
import { StreamInfoDisplay } from 'ui/components/stream-info/stream-info-display';
import { gamePath } from 'settings';
import { PLAYER_TWITCH_EVERYWHERE, PLAYER_IMDB } from 'util/player-type';
import { TwitchEverywherePanelContainer } from 'ui/containers/twitch-everywhere-panel';
import { WatchPartyStreamInfo } from 'ui/components/stream-info/watch-party';
import { AdContentTypes } from 'actions/ads';
import { SIDEBAR_VIEW } from 'state/collection';

const DEFAULT_ARGS = Object.freeze({
    adPlaying: false,
    channelName: '',
    displayName: '',
    gameName: '',
    lang: {},
    logo: '',
    name: '',
    isMobile: false,
    status: '',
    streamUrl: '',
    viewerCount: 0,
    pause: () => {}, // this gets defined on connect() but we don't call that in our tests, so use a dummy function
    showInfo: true,
});

const defaultState = Object.freeze({
    ads: {
        currentMetadata: {
            contentType: '',
        },
    },
    env: {
        playerType: '',
    },
    lang: {},
    playerOptions: {
        showInfo: true,
    },
    streamMetadata: {
        channelName: '',
        displayName: '',
        game: '',
        streamType: '',
        channel: {
            logo: '',
            status: '',
        },
        name: '',
        url: '',
    },
    viewercount: '',
    watchParty: {
        vodId: '',
        title: '',
        viewable: '',
        broadcastType: '',
    },
    collection: {
        currentView: '',
    },
});

function renderStreamInfoComponent(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <StreamInfoComponent {...args} />;
    return shallow(component);
}

describe('ui | containers | top-bar | stream-info', function() {
    test('given default input, should create empty div', function() {
        const streamInfo = renderStreamInfoComponent();
        expect(streamInfo.type()).toBe('div', 'should create a div');
        expect(streamInfo.children().length).toBe(0, 'should have no children');
    });

    test('given logo input, should create logo component', function() {
        const input = {
            logo: 'https://logo.url/',
            streamUrl: 'https://stream.url/',
        };
        const streamInfo = renderStreamInfoComponent(input);
        const logoClassNames = {
            a: 'player-streaminfo__picture',
            img: 'qa-broadcaster-logo',
        };
        const logoTarget = '_blank';

        expect(streamInfo.type()).toBe('div', 'should create a div');
        expect(streamInfo.children().length).toBe(1, 'should have 1 child');
        expect(streamInfo.containsMatchingElement(
            <LinkedImage
                classNames={logoClassNames}
                imgSrc={input.logo}
                href={input.streamUrl}
                onClick={streamInfo.instance().props.pause}
                target={logoTarget}
            />
        )).toBeTruthy();
    });

    test('given live input, should create a live streamInfoDisplay component,', function() {
        const input = {
            channelName: 'channelName', // channelName indicates liveness
            displayName: 'displayName',
            name: 'streamName',
            status: 'live status',
            gameName: 'gameName',
            streamUrl: 'https://stream.url/',
            viewerCount: 12345,
            lang: {},
            isMobile: false,
        };
        const streamInfo = renderStreamInfoComponent(input);
        const gameUrl = `${gamePath}/${encodeURIComponent(input.gameName)}`;

        expect(streamInfo.type()).toBe('div', 'should create a div');
        expect(streamInfo.children().length).toBe(1, 'should have 1 child');
        expect(streamInfo.containsMatchingElement(
            <StreamInfoDisplay
                live={true}
                name={input.displayName}
                title={input.status}
                gameName={input.gameName}
                gameUrl={gameUrl}
                onClick={streamInfo.instance().props.pause}
                streamUrl={input.streamUrl}
                viewerCount={input.viewerCount}
                lang={input.lang}
                isMobile={input.isMobile}
            />
        )).toBeTruthy();
    });

    test('given vod input, should create a vod streamInfoDisplay component,', function() {
        const input = {
            displayName: 'displayName',
            name: 'streamName',
            status: 'live status',
            gameName: 'gameName',
            streamUrl: 'https://stream.url/',
            viewerCount: 12345,
            isMobile: true,
            lang: {},
        };
        const streamInfo = renderStreamInfoComponent(input);
        const gameUrl = `${gamePath}/${encodeURIComponent(input.gameName)}`;

        expect(streamInfo.type()).toBe('div', 'should create a div');
        expect(streamInfo.children().length).toBe(1, 'should have 1 child');
        expect(streamInfo.containsMatchingElement(
            <StreamInfoDisplay
                live={false}
                name={input.displayName}
                title={input.name}
                gameName={input.gameName}
                gameUrl={gameUrl}
                streamUrl={input.streamUrl}
                viewerCount={input.viewerCount}
                lang={input.lang}
                isMobile={input.isMobile}
            />
        )).toBeTruthy();
    });

    test('given the IMDB playerType, renders null', function() {
        const streamInfo = renderStreamInfoComponent({
            playerType: PLAYER_IMDB,
        });

        expect(streamInfo.type()).toBe(null);
    });

    test('given that adPlaying is true, renders null', function() {
        const streamInfo = renderStreamInfoComponent({
            adPlaying: true,
        });

        expect(streamInfo.type()).toBe(null);
    });

    test('given that showInfo is false, renders null', function() {
        const streamInfo = renderStreamInfoComponent({
            showInfo: false,
        });

        expect(streamInfo.type()).toBe(null);
    });

    // eslint-disable-next-line max-len
    test('given the Twitch Everywhere player type, renders the TwitchEverywherePanelContainer', function() {
        const streamInfo = renderStreamInfoComponent({
            playerType: PLAYER_TWITCH_EVERYWHERE,
        });

        expect(streamInfo.contains(<TwitchEverywherePanelContainer />)).toBeTruthy();
    });

    // eslint-disable-next-line max-len
    test('given a watch party data, should render watch party component', function() {
        const args = {
            isWatchParty: true,
            vodId: '12345',
            vodTitle: 'Test Title',
            vodIsViewable: true,
            vodBroadcastType: 'highlight',
            lang: { translate: s => s },
        };
        const streamInfo = renderStreamInfoComponent(args);
        expect(
            streamInfo.contains(<WatchPartyStreamInfo {...assign({}, DEFAULT_ARGS, args)} />)
        ).toBeTruthy();
    });

    test('maps adPlaying to true if not NONE', function() {
        const state = assign({}, defaultState, {
            ads: {
                currentMetadata: {
                    contentType: AdContentTypes.IMA,
                },
            },
        });
        const mappedState = mapStateToProps(state);
        expect(mappedState.adPlaying).toBe(true, 'adPlaying should be true');
    });

    test('maps adPlaying to false if NONE', function() {
        const state = assign({}, defaultState, {
            ads: {
                currentMetadata: {
                    contentType: AdContentTypes.NONE,
                },
            },
        });
        const mappedState = mapStateToProps(state);
        expect(mappedState.adPlaying).toBe(false, 'adPlaying should be false');
    });

    test('maps showInfo to false if collection sidebar is open', () => {
        const state = assign({}, defaultState, {
            collection: {
                currentView: SIDEBAR_VIEW,
            },
        });
        const mappedState = mapStateToProps(state);
        expect(mappedState.showInfo).toBe(false, 'showInfo should be false');
    });

    test('maps showInfo to true if showInfo is true', function() {
        const state = assign({}, defaultState, {
            playerOptions: {
                showInfo: true,
            },
        });
        const mappedState = mapStateToProps(state);
        expect(mappedState.showInfo).toBe(true, 'showInfo should be true');
    });

    test('maps showInfo to true if showInfo is false and isFullScreen is true', function() {
        const state = assign({}, defaultState, {
            playerOptions: {
                showInfo: false,
            },
            screenMode: {
                isFullScreen: true,
            },
        });
        const mappedState = mapStateToProps(state);
        expect(mappedState.showInfo).toBe(true, 'showInfo should be true');
    });

    test('maps showInfo to true if showInfo is false and isTheatreMode is true', function() {
        const state = assign({}, defaultState, {
            playerOptions: {
                showInfo: false,
            },
            screenMode: {
                isTheatreMode: true,
            },
        });
        const mappedState = mapStateToProps(state);
        expect(mappedState.showInfo).toBe(true, 'showInfo should be true');
    });
});
