import React from 'react';
import assign from 'lodash/assign';
import { shallow } from 'enzyme';
import { AnimatedUnmuteOverlay } from 'ui/components/overlays/unmute-overlay';
import {
    UnmuteOverlayContainer,
    mapStateToProps,
    mapDispatchToProps,
} from 'ui/containers/overlays/unmute-overlay';
import { ACTION_VIDEO_API_MUTE_PLAYER } from 'actions/video-api';
import { ACTION_TRACK_EVENT } from 'actions/analytics-tracker';
import { ACTION_PUSH_SCREEN, ACTION_POP_SCREEN, UNMUTE_SCREEN,
         CONTENT_SCREEN } from 'actions/screen';
import TransitionGroup from 'react-transition-group/TransitionGroup';

const DEFAULT_PROPS = Object.freeze({
    unmutePlayer: () => {},
    autoplayEnabled: true,
    isMobileWeb: true,
    isPlaying: false,
    hasPlayed: false,
    isMuted: false,
    setTimeout: () => {},
    clearTimeout: () => {},
    trackOverlayClick: () => {},
    pushUnmuteScreen: () => {},
    popScreen: () => {},
    screenStack: [UNMUTE_SCREEN, CONTENT_SCREEN],
});

function renderContainer(overrides = {}) {
    const args = assign({}, DEFAULT_PROPS, overrides);
    const component = <UnmuteOverlayContainer {...args} />;
    return shallow(component);
}

describe('ui | containers | overlays | unmute-overlays', () => {
    test('correctly maps state to props', () => {
        const state = {
            playback: {
                autoplay: true,
                playing: true,
                hasPlayed: true,
                muted: true,
            },
            window: {
                clearTimeout: () => {},
                setTimeout: () => {},
            },
            env: {
                isMobileLocation: true,
            },
            screen: [],
        };

        expect(mapStateToProps(state)).toEqual({
            autoplayEnabled: state.playback.autoplay,
            isMobileWeb: state.env.isMobileLocation,
            isPlaying: state.playback.playing,
            hasPlayed: state.playback.hasPlayed,
            isMuted: state.playback.muted,
            setTimeout: state.window.setTimeout,
            clearTimeout: state.window.clearTimeout,
            screenStack: state.screen,
        });
    });

    describe('correctly maps dispatches to props', () => {
        let dispatchSpy;
        let dispatchProps;

        beforeEach(() => {
            dispatchSpy = jest.fn();
            dispatchProps = mapDispatchToProps(dispatchSpy);
        });

        test('unmutePlayer', () => {
            dispatchProps.unmutePlayer();
            expect(dispatchSpy.mock.calls[0][0].type).toBe(ACTION_VIDEO_API_MUTE_PLAYER);
            expect(dispatchSpy.mock.calls[0][0].muted).toBe(false);
            expect(dispatchSpy.mock.calls[0][0].automated).toBe(false);
        });

        test('trackOverlayClick', () => {
            dispatchProps.trackOverlayClick();
            expect(dispatchSpy.mock.calls[0][0].type).toBe(ACTION_TRACK_EVENT);
            expect(dispatchSpy.mock.calls[0][0].eventName).toBe('player_click');
            expect(dispatchSpy.mock.calls[0][0].eventProperties).toEqual({
                component: 'unmute_overlay',
            });
        });

        test('pushUnmuteScreen', () => {
            dispatchProps.pushUnmuteScreen();
            expect(dispatchSpy.mock.calls[0][0].type).toBe(ACTION_PUSH_SCREEN);
            expect(dispatchSpy.mock.calls[0][0].screen).toBe(UNMUTE_SCREEN);
        });

        test('popScreen', () => {
            dispatchProps.popScreen();
            expect(dispatchSpy.mock.calls[0][0].type).toBe(ACTION_POP_SCREEN);
        });
    });

    test('the timeout handler pops the screen stack if the unmute screen is on top', () => {
        const popScreenSpy = jest.fn();
        const container = renderContainer({
            popScreen: popScreenSpy,
        });
        const timeoutHandler = container.instance().timeoutHandler;

        container.setProps({ screenStack: [UNMUTE_SCREEN, CONTENT_SCREEN] });
        popScreenSpy.mockReset();

        timeoutHandler();
        expect(popScreenSpy.mock.calls.length).toBe(1);
    });

    test('the timeout handler does not pop the screen stack if the unmute screen is not on top', () => {
        const popScreenSpy = jest.fn();
        const container = renderContainer({
            popScreen: popScreenSpy,
        });
        const timeoutHandler = container.instance().timeoutHandler;

        container.setProps({ screenStack: [CONTENT_SCREEN] });
        popScreenSpy.mockReset();

        timeoutHandler();
        expect(popScreenSpy.mock.calls.length).toBe(0);
    });

    // eslint-disable-next-line max-len
    test('on unmute screen being on top of the screenstack, clear any old timeout, start a timeout to pop unmute screen, and set timeout id in state', () => {
        const oldTimeoutId = 1234567;
        const newTimeoutId = 123121424;
        const setTimeoutSpy = jest.fn();
        const clearTimeoutSpy = jest.fn();
        const container = renderContainer({
            setTimeout: setTimeoutSpy,
            clearTimeout: clearTimeoutSpy,
            screenStack: [CONTENT_SCREEN],
        });
        const timeoutHandler = container.instance().timeoutHandler;

        container.setState({
            hideTimeoutId: oldTimeoutId,
        });

        clearTimeoutSpy.mockReset();
        setTimeoutSpy.mockReset();
        setTimeoutSpy.mockReturnValue(newTimeoutId);

        container.setProps({
            screenStack: [UNMUTE_SCREEN, CONTENT_SCREEN],
        });

        expect(clearTimeoutSpy.mock.calls[0][0]).toBe(oldTimeoutId);
        expect(setTimeoutSpy.mock.calls[0][0]).toBe(timeoutHandler);
        expect(setTimeoutSpy.mock.calls[0][1]).toBe(5000);
        expect(container.state('hideTimeoutId')).toBe(newTimeoutId);
    });

    test('on unmute screen not being on top of the screenstack, clear timeout id in state', () => {
        const oldTimeoutId = 1234567;
        const clearTimeoutSpy = jest.fn();
        const container = renderContainer({
            clearTimeout: clearTimeoutSpy,
            screenStack: [UNMUTE_SCREEN, CONTENT_SCREEN],
        });

        container.setState({
            hideTimeoutId: oldTimeoutId,
        });

        clearTimeoutSpy.mockReset();

        container.setProps({
            screenStack: [CONTENT_SCREEN],
        });

        expect(clearTimeoutSpy.mock.calls[0][0]).toBe(oldTimeoutId);
        expect(container.state('hideTimeoutId')).toBe(0);
    });

    // eslint-disable-next-line max-len
    test('renders transitiongroup with animatedUnmuteOverlay child when unmute screen is on top of screen stack', () => {
        const component = renderContainer();
        component.setProps({ screenStack: [UNMUTE_SCREEN, CONTENT_SCREEN] });
        expect(component.type()).toBe(TransitionGroup);
        expect(component.childAt(0).type()).toBe(AnimatedUnmuteOverlay);
    });

    test('renders only transitiongroup when unmute screen is not on top of screen stack', () => {
        const component = renderContainer();
        component.setProps({ screenStack: [CONTENT_SCREEN] });
        expect(component.type()).toBe(TransitionGroup);
        expect(component.children().length).toBe(0);
    });

    test('animatedUnmuteOverlay onClick handler calls props.unmutePlayer', () => {
        const unmutePlayerSpy = jest.fn();
        const component = renderContainer({
            unmutePlayer: unmutePlayerSpy,
            screenStack: [UNMUTE_SCREEN, CONTENT_SCREEN],
        });
        const unmuteOverlay = component.childAt(0);
        const onClickHandler = unmuteOverlay.prop('onClick');

        unmutePlayerSpy.mockReset();

        onClickHandler();

        expect(unmutePlayerSpy.mock.calls.length).toBe(1);
    });

    test('animatedUnmuteOverlay onClick handler calls props.trackOverlayClick', () => {
        const trackOverlayClickSpy = jest.fn();
        const component = renderContainer({
            trackOverlayClick: trackOverlayClickSpy,
            screenStack: [UNMUTE_SCREEN, CONTENT_SCREEN],
        });
        const unmuteOverlay = component.childAt(0);
        const onClickHandler = unmuteOverlay.prop('onClick');

        trackOverlayClickSpy.mockReset();

        onClickHandler();

        expect(trackOverlayClickSpy.mock.calls.length).toBe(1);
    });
});
