import { ClipsBenchmarkTracking } from 'ui/player-types/clips/utils/tracking/clips-benchmark-tracking';
import {
    TRACKING_PAGEVIEW_EVENT,
    TRACKING_BENCHMARK_FETCH_START,
    TRACKING_BENCHMARK_COMPLETE_TRANSITION,
    TRACKING_BENCHMARK_VIDEO_STARTED,
} from 'ui/player-types/clips/utils/tracking/clips-tracking-constants';
import assign from 'lodash/assign';

describe('ui | player-types | clips | utils | tracking | clips-benchmark-tracking', () => {
    let store;
    let state;
    let clipsTrackEventSpy;
    let clipsBenchmarkTracking;

    beforeEach(() => {
        clipsTrackEventSpy = jest.fn();

        state = {
            analyticsTracker: {
                clipsTrackEvent: clipsTrackEventSpy,
            },
            playback: {
                hasPlayed: false,
            },
            streamMetadata: {
                url: null,
            },
            window: {
                Benchmark: 123,
            },
        };

        store = {
            getState: () => state,
            subscribe() {},
        };

        clipsBenchmarkTracking = new ClipsBenchmarkTracking(store);
        clipsBenchmarkTracking.shouldSendEvents = true;
        clipsBenchmarkTracking.currentTime = () => 5000;
    });

    describe('ClipsBenchmarkTracking', () => {
        test('should call trackFetchStart on init', () => {
            expect(clipsBenchmarkTracking.hasTrackedEventMap).toEqual({
                [TRACKING_PAGEVIEW_EVENT]: false,
                [TRACKING_BENCHMARK_FETCH_START]: true,
                [TRACKING_BENCHMARK_COMPLETE_TRANSITION]: false,
                [TRACKING_BENCHMARK_VIDEO_STARTED]: false,
            });
        });

        describe('trackPageview', () => {
            test('should call it only once', () => {
                const shouldCallState = assign(
                    {},
                    state,
                    {
                        streamMetadata: {
                            url: 'some-url',
                        },
                    }
                );
                expect(clipsTrackEventSpy).toHaveBeenCalledTimes(0);
                clipsBenchmarkTracking.handleStateChange(shouldCallState);

                expect(clipsTrackEventSpy).toHaveBeenCalledTimes(1);
                // eslint-disable-next-line camelcase
                expect(clipsTrackEventSpy).toHaveBeenCalledWith(TRACKING_PAGEVIEW_EVENT, { client_time: 5 });

                clipsBenchmarkTracking.handleStateChange(shouldCallState);
                expect(clipsTrackEventSpy).toHaveBeenCalledTimes(1);
            });
        });

        describe('trackVideoPlaybackStarted', () => {
            test('should call it only once', () => {
                const shouldCallState = assign(
                    {},
                    state,
                    {
                        playback: {
                            hasPlayed: true,
                        },
                    }
                );
                expect(clipsTrackEventSpy).toHaveBeenCalledTimes(0);
                clipsBenchmarkTracking.handleStateChange(shouldCallState);

                expect(clipsTrackEventSpy).toHaveBeenCalledTimes(1);
                // eslint-disable-next-line camelcase
                expect(clipsTrackEventSpy).toHaveBeenCalledWith(TRACKING_BENCHMARK_VIDEO_STARTED, { client_time: 5 });

                clipsBenchmarkTracking.handleStateChange(shouldCallState);
                expect(clipsTrackEventSpy).toHaveBeenCalledTimes(1);
            });
        });

        describe('trackAppReady', () => {
            test('should call tracking event and set the proper event name to true', () => {
                const time = 1000;
                clipsBenchmarkTracking.timeFromFetchStartMillis = () => time;

                expect(clipsTrackEventSpy).toHaveBeenCalledTimes(0);
                clipsBenchmarkTracking.trackAppReady();

                expect(clipsTrackEventSpy).toHaveBeenCalledTimes(1);
                expect(clipsTrackEventSpy).toHaveBeenCalledWith(
                    TRACKING_BENCHMARK_COMPLETE_TRANSITION,
                    {
                        destination: 'clips',
                        // eslint-disable-next-line camelcase
                        time_from_fetch: time,
                        // eslint-disable-next-line camelcase
                        client_time: 5,
                    }
                );
            });
        });
    });
});
