import { ClipsLoadingTimeSpentTracker } from 'ui/player-types/clips/utils/tracking/clips-loading-time-spent-tracker';
import {
    TRACKING_LOCATION_CLIPS,
    TRACKING_CLIP_FORMAT,
    TRACKING_CLIPS_QOS,
    TRACKING_CLIPS_BUFFER_REFILL,
} from 'ui/player-types/clips/utils/tracking/clips-tracking-constants';
import assign from 'lodash/assign';

describe('ui | player-types | clips | utils | tracking | clips-loading-time-spent-tracker', () => {
    let store;
    let state;
    let clipsTrackEventSpy;
    let clipsLoadingTimeSpent;

    beforeEach(() => {
        clipsTrackEventSpy = jest.fn();

        state = {
            analyticsTracker: {
                clipsTrackEvent: clipsTrackEventSpy,
            },
            playback: {
                playing: false,
            },
            quality: {
                selected: 1,
            },
        };

        store = {
            getState: () => state,
            subscribe() {},
            dispatch() {},
        };

        const player = {
            addEventListener: jest.fn(),
        };

        clipsLoadingTimeSpent = new ClipsLoadingTimeSpentTracker(store, player);
    });

    describe('ClipsLoadingTimeSpentTracker', () => {
        test('initializes properly', () => {
            expect(clipsLoadingTimeSpent.videoPlayedOnceThrough).toEqual(false);
        });

        test('when playback ends videoPlayedOnceThrough becomes true', () => {
            expect(clipsLoadingTimeSpent.videoPlayedOnceThrough).toEqual(false);

            clipsLoadingTimeSpent.handleStateChange(
                assign(
                    {},
                    state,
                    {
                        playback: {
                            ended: true,
                        },
                    }
                ),
                state
            );

            expect(clipsLoadingTimeSpent.videoPlayedOnceThrough).toEqual(true);
        });

        describe('trackBufferEmpty', () => {
            beforeEach(() => {
                clipsLoadingTimeSpent.trackBufferEmptyIfStalled = jest.fn();
                clipsLoadingTimeSpent.trackBufferEmpty = jest.fn();
            });

            test('calls trackBufferEmpty when the quality has changed', () => {
                clipsLoadingTimeSpent.videoPlayedOnceThrough = false;

                clipsLoadingTimeSpent.handleStateChange(
                    assign(
                        {},
                        state,
                        {
                            playback: {
                                hasPlayed: true,
                                isLoading: true,
                            },
                            quality: {
                                selected: 2,
                            },
                        }
                    ),
                    state
                );

                expect(clipsLoadingTimeSpent.trackBufferEmpty).toHaveBeenCalledTimes(1);
            });

            test('calls trackBufferEmptyIfStalled when the video is paused and loading', () => {
                clipsLoadingTimeSpent.videoPlayedOnceThrough = false;

                clipsLoadingTimeSpent.handleStateChange(
                    assign(
                        {},
                        state,
                        {
                            playback: {
                                hasPlayed: true,
                                isLoading: true,
                            },
                        }
                    ),
                    state
                );

                expect(clipsLoadingTimeSpent.trackBufferEmptyIfStalled).toHaveBeenCalledTimes(1);
            });

            test('does not call trackBufferEmpty when the video has not played yet', () => {
                clipsLoadingTimeSpent.videoPlayedOnceThrough = false;

                clipsLoadingTimeSpent.handleStateChange(
                    assign(
                        {},
                        state,
                        {
                            playback: {
                                hasPlayed: false,
                                isLoading: true,
                            },
                        }
                    ),
                    state
                );

                expect(clipsLoadingTimeSpent.trackBufferEmpty).toHaveBeenCalledTimes(0);
                expect(clipsLoadingTimeSpent.trackBufferEmptyIfStalled).toHaveBeenCalledTimes(0);
            });

            test('does not call trackBufferEmpty when the video has played through', () => {
                clipsLoadingTimeSpent.videoPlayedOnceThrough = true;

                clipsLoadingTimeSpent.handleStateChange(
                    assign(
                        {},
                        state,
                        {
                            playback: {
                                hasPlayed: true,
                                isLoading: true,
                            },
                        }
                    ),
                    state
                );

                expect(clipsLoadingTimeSpent.trackBufferEmpty).toHaveBeenCalledTimes(0);
                expect(clipsLoadingTimeSpent.trackBufferEmptyIfStalled).toHaveBeenCalledTimes(0);
            });
        });

        describe('trackBufferRefill', () => {
            test('calls trackBufferRefill when the video stops loading again', () => {
                clipsLoadingTimeSpent.bufferEmptyStartTime = 1600;
                clipsLoadingTimeSpent.currentTime = () => 3000;

                clipsLoadingTimeSpent.handleStateChange(
                    assign(
                        {},
                        state,
                        {
                            playback: {
                                hasPlayed: true,
                                isLoading: false,
                            },
                        }
                    ),
                    state
                );

                expect(clipsTrackEventSpy).toHaveBeenCalledWith(
                    TRACKING_CLIPS_QOS,
                    {
                        /* eslint-disable camelcase */
                        backend: TRACKING_LOCATION_CLIPS,
                        vod_format: TRACKING_CLIP_FORMAT,
                        type: TRACKING_CLIPS_BUFFER_REFILL,
                        buffering_time: 1.4,
                        /* eslint-enable camelcase */
                    }
                );
            });

            test('does not call trackBufferRefill when there is no buffer empty time', () => {
                clipsLoadingTimeSpent.bufferEmptyStartTime = null;

                clipsLoadingTimeSpent.handleStateChange(
                    assign(
                        {},
                        state,
                        {
                            playback: {
                                hasPlayed: true,
                                isLoading: false,
                            },
                        }
                    ),
                    state
                );

                expect(clipsTrackEventSpy).toHaveBeenCalledTimes(0);
            });
        });
    });
});
