import { Video } from 'video';
import { init as initStore } from 'state';
import { CONTENT_MODE_LIVE } from 'stream/twitch-live';
import { CONTENT_MODE_VOD } from 'stream/twitch-vod';
import { CONTENT_MODE_CLIP } from 'stream/clip';
import { ACTION_CREATE_ADS_MANAGER } from 'actions/ads-manager';
import { ACTION_IMA_SCRIPT_LOADED } from 'actions/ads';
import { ACTION_SET_LOADING, ACTION_PLAYING, ACTION_WAITING,
         ACTION_PAUSE, ACTION_ENDED, ACTION_INCREMENT_BUFFER_EMPTIES,
         ACTION_PLAYER_SEEKING, ACTION_PLAYER_SEEKED, ACTION_SET_START_TIME } from 'actions/playback';
import { selectQuality, KEY_AUTO_QUALITY_FORCED, QUALITY_AUTO,
         setCurrentQuality } from 'actions/quality';
import { setOnline, ACTION_SET_ONLINE } from 'actions/online';
import { ACTION_SET_PERSISTENCE } from 'actions/settings';
import { videoAPILoaded } from 'actions/video-api';
import { localStore } from 'util/storage';
import each from 'lodash/each';
import * as TwitchEvents from 'backend/events/twitch-event';
import * as MediaEvents from 'backend/events/media-event';
import * as AdEvents from 'ads/advertisement-event';
import { createMockQualities } from 'test-utils/fakes/qualities';
import { ACTION_USE_EVENT_EMITTER } from 'actions/event-emitter';
import { setAnalyticsTracker } from 'actions/analytics-tracker';
import { ExtensionCoordinator } from 'extension-coordinator';
import EventEmitter from 'event-emitter';
import { PLAYBACK_ERROR } from 'analytics/spade-events';
import * as BackendUtils from 'backend/util';
import { BACKEND_PLAYER_CORE } from 'backend/player-core';
import { BACKEND_FLASH } from 'backend/flash';
import { BACKEND_BLANK_TYPE } from 'backend/blank';
import { BACKEND_MEDIA_PLAYER } from 'backend/mediaplayer';
import { BACKEND_HLS } from 'backend/hls';
import Errors from 'errors';

jest.mock('backend/flash', function() {
    return require('../../mocks/backend/flash');
});

jest.mock('backend/mediaplayer', function() {
    return require('../../mocks/backend/mediaplayer');
});

jest.mock('backend/hls', function() {
    return require('../../mocks/backend/hls');
});

jest.mock('backend/blank', function() {
    return require('../../mocks/backend/blank-backend');
});

jest.mock('backend/player-core', function() {
    return require('../../mocks/backend/player-core');
});

// Events for loading logic
const showLoadingEvents = [
    TwitchEvents.MANIFEST_EXTRA_INFO,
    MediaEvents.WAITING,
    MediaEvents.SEEKING,
];
const hideLoadingEvents = [
    MediaEvents.PLAYING,
];

describe('video', () => {
    const rootElm = document.createElement('div');

    describe('Clip Content', () => {
        let video;
        let backend;
        let store;

        beforeEach(() => {
            store = initStore({
                stream: {
                    contentType: CONTENT_MODE_CLIP,
                },
            });

            video = new Video(rootElm, store, {
                backend: 'mediaplayer',
            });
            backend = store.getState().backend;
        });

        afterEach(() => {
            video.destroy();
            backend.destroy();
            store = null;
        });

        describe('on LOADED_METADATA event from backend', () => {
            test('seek with startTime in store', () => {
                const startTime = 10;

                store.dispatch({
                    type: ACTION_SET_START_TIME,
                    startTime,
                });

                jest.spyOn(backend, 'setCurrentTime');
                backend._events.emit(MediaEvents.LOADED_METADATA);

                expect(backend.setCurrentTime).toHaveBeenCalledTimes(1);
                expect(backend.setCurrentTime.mock.calls[0][0]).toBe(startTime);
            });
        });
    });

    describe('For live content', () => {
        let video;
        let backend;
        let store;

        beforeEach(() => {
            store = initStore({
                stream: {
                    contentType: CONTENT_MODE_LIVE,
                    restrictedBitrates: [],
                },
            });

            video = new Video(rootElm, store, {
                backend: 'mediaplayer',
            });
            backend = store.getState().backend;
        });

        afterEach(() => {
            video.destroy();
            backend.destroy();
            store = null;
        });

        describe('emits event', () => {
            test('online when state property online changes to true', () => {
                store.dispatch(setOnline(false));
                const onlineEventHandler = jest.fn();
                const offlineEventHandler = jest.fn();

                video.addEventListener(TwitchEvents.ONLINE, onlineEventHandler);
                video.addEventListener(TwitchEvents.OFFLINE, offlineEventHandler);

                store.dispatch(setOnline(true));
                expect(onlineEventHandler).toHaveBeenCalledTimes(1);
                expect(offlineEventHandler).toHaveBeenCalledTimes(0);
            });

            test('offline when state property online changes to false', () => {
                store.dispatch(setOnline(true));
                const onlineEventHandler = jest.fn();
                const offlineEventHandler = jest.fn();

                video.addEventListener(TwitchEvents.ONLINE, onlineEventHandler);
                video.addEventListener(TwitchEvents.OFFLINE, offlineEventHandler);

                store.dispatch(setOnline(false));
                expect(onlineEventHandler).toHaveBeenCalledTimes(0);
                expect(offlineEventHandler).toHaveBeenCalledTimes(1);
            });
        });

        describe('Test when ABS support is available', () => {
            test('For first time user auto should be forced', () => {
                localStore.set(KEY_AUTO_QUALITY_FORCED, false);
                jest.spyOn(video, 'setQuality');
                backend._events.emit(TwitchEvents.MANIFEST_EXTRA_INFO, {});

                const setQualityArgs = video.setQuality.mock.calls[0][0];
                expect(setQualityArgs).toBe(QUALITY_AUTO);
                const autoForced = localStore.get(KEY_AUTO_QUALITY_FORCED, false);
                expect(autoForced).toBe(true);
            });
        });

        describe('Test when ABS support is unavailable', () => {
            test('The default quality picked should be medium', () => {
                jest.spyOn(video, 'setQuality');
                backend._events.emit(TwitchEvents.MANIFEST_EXTRA_INFO, {});
                const setQualityArgs = video.setQuality.mock.calls[0][0];
                expect(setQualityArgs).toBe('medium');
                const autoForced = localStore.get(KEY_AUTO_QUALITY_FORCED, false);
                expect(autoForced).toBe(true);
            });
        });
    });

    describe('For VOD content', () => {
        let video;
        let backend;
        let store;

        beforeEach(() => {
            store = initStore({
                stream: {
                    contentType: CONTENT_MODE_VOD,
                    restrictedBitrates: [],
                },
            });

            video = new Video(rootElm, store, {
                backend: 'mediaplayer',
            });
            backend = store.getState().backend;
        });

        afterEach(() => {
            video.destroy();
            backend.destroy();
            store = null;
        });

        describe('on LOADED_METADATA event from backend', () => {
            test('seek with startTime in store', () => {
                const startTime = 10;

                store.dispatch({
                    type: ACTION_SET_START_TIME,
                    startTime,
                });

                jest.spyOn(backend, 'setCurrentTime');
                backend._events.emit(MediaEvents.LOADED_METADATA);

                expect(backend.setCurrentTime).toHaveBeenCalledTimes(1);
                expect(backend.setCurrentTime.mock.calls[0][0]).toBe(startTime);
            });
        });

        describe('Test ABS support for VOD', () => {
            test('Quality set should be auto', () => {
                localStore.set(KEY_AUTO_QUALITY_FORCED, false);
                jest.spyOn(video, 'setQuality');
                backend._events.emit(TwitchEvents.MANIFEST_EXTRA_INFO, {});

                const setQualityArgs = video.setQuality.mock.calls[0][0];
                expect(setQualityArgs).toBe(QUALITY_AUTO);
                const autoForced = localStore.get(KEY_AUTO_QUALITY_FORCED, false);
                expect(autoForced).toBe(true);
            });
        });
    });

    describe('For any content', () => {
        let video;
        let backend;
        let store;

        beforeEach(() => {
            store = initStore({
                stream: {
                    contentType: CONTENT_MODE_LIVE,
                    restrictedBitrates: [],
                },
            });

            video = new Video(rootElm, store, {
                backend: 'mediaplayer',
            });
            backend = store.getState().backend;

            jest.spyOn(store, 'dispatch');
        });

        afterEach(() => {
            video.destroy();
            backend.destroy();
            store = null;
        });

        test('creates and loads an event emitter', () => {
            new Video(rootElm, store, {
                backend: 'player-core',
            });

            expect(store.dispatch).toHaveBeenCalled();

            const firstCallArgs = store.dispatch.mock.calls[0][0];

            expect(firstCallArgs.type).toBe(ACTION_USE_EVENT_EMITTER);
            expect(firstCallArgs.emitter instanceof EventEmitter).toBe(true);
        });

        test('setTheatre', () => {
            video.setTheatre(false);
            expect(video.getTheatre()).toBe(false);

            const theatreChangeListener = jest.fn();
            video.addEventListener(TwitchEvents.THEATRE_CHANGE, theatreChangeListener);

            video.setTheatre(true);
            expect(video.getTheatre()).toBe(true);

            expect(theatreChangeListener).toHaveBeenCalledTimes(1);
        });

        test('isPaused', () => {
            const getPausedResult = video.getPaused();
            const isPausedResult = video.isPaused();
            expect(isPausedResult).toBe(getPausedResult);
        });

        test('setCurrentTime', () => {
            const currentTime = 100;
            const action = {
                type: ACTION_PLAYER_SEEKING,
            };
            jest.spyOn(backend, 'setCurrentTime');
            video.setCurrentTime(currentTime);
            expect(store.dispatch.mock.calls[0][0]).toEqual(action);
            expect(backend.setCurrentTime.mock.calls[0][0]).toBe(currentTime);
        });

        test('trackMiniPlayerAction', () => {
            const fakeTracker = { trackEvent: jest.fn() };
            store.dispatch(setAnalyticsTracker(fakeTracker));
            video.trackMiniPlayerAction('close', 'user_close');

            const [eventName, action] = fakeTracker.trackEvent.mock.calls[0];
            expect(eventName).toBe('site_mini_player_action');
            expect(action).toEqual({
                action: 'close',
                reason: 'user_close',
            });
            expect(fakeTracker.trackEvent).toHaveBeenCalledTimes(1);
        });

        describe('on init', () => {
            test('it calls registerPlayer on the ExtensionService with itself', () => {
                const { registerPlayer } = ExtensionCoordinator.ExtensionService;
                expect(registerPlayer).toHaveBeenLastCalledWith(video);
            });

            test('it listens for context updates', () => {
                const { listenForContext } = ExtensionCoordinator.ExtensionService;
                expect(listenForContext).toHaveBeenCalled();
            });
        });

        describe('Ad Manager Event Handlers', () => {
            test('bubbles up AD_* event from ad manager when its updated in statestore', () => {
                const _events = {};
                const adsManager = {
                    addEventListener(event, cb) {
                        _events[event] = _events[event] ?
                            _events[event].concat(cb) :
                            [cb];
                    },
                    _emit(event) {
                        _events[event].forEach(cb => cb({}));
                    },
                };

                store.dispatch({
                    type: ACTION_IMA_SCRIPT_LOADED,
                    imaScriptLoaded: true,
                });
                store.dispatch({
                    type: ACTION_CREATE_ADS_MANAGER,
                    adsManager,
                });

                each(AdEvents, adEvent => {
                    const videoListener = jest.fn();
                    const { adsManager } = store.getState();
                    video.addEventListener(adEvent, videoListener);

                    adsManager._emit(adEvent);

                    expect(videoListener).toHaveBeenCalledTimes(1);
                    video.removeEventListener(adEvent, videoListener);
                });
            });
        });

        describe('Backend Event Handlers', () => {
            test('on player_init, load video api middleware', () => {
                backend._events.emit(TwitchEvents.PLAYER_INIT);
                const resultAction = store.dispatch.mock.calls[0][0];
                expect(resultAction).toEqual(videoAPILoaded(video));
            });

            function checkEventPropagation(event, payload = {}) {
                test(`${event} should propagate from the backend once`, () => {
                    const callbackSpy = jest.fn();
                    video.addEventListener(event, callbackSpy);
                    backend._events.emit(event, payload);
                    expect(callbackSpy).toHaveBeenCalledTimes(1);
                });
            }
            // Every event listened to in video.js and fired by player-core
            const NO_PAYLOAD_EVENTS = [
                MediaEvents.CAN_PLAY,
                MediaEvents.ENDED,
                MediaEvents.LOADED_METADATA,
                MediaEvents.LOADSTART,
                MediaEvents.PAUSE,
                MediaEvents.PLAYING,
                MediaEvents.SEEKING,
                MediaEvents.TIME_UPDATE,
                MediaEvents.WAITING,
                TwitchEvents.BUFFER_CHANGE,
                TwitchEvents.PLAYER_INIT,
                MediaEvents.DURATION_CHANGE,
                MediaEvents.PLAY,
                TwitchEvents.RESTRICTED,
                TwitchEvents.SEGMENT_CHANGE,
                MediaEvents.SEEKED,
            ];

            NO_PAYLOAD_EVENTS.forEach(checkEventPropagation);

            checkEventPropagation(TwitchEvents.ABS_STREAM_FORMAT_CHANGE, {
                stream_format_current: 'medium', // eslint-disable-line camelcase
                stream_format_previous: 'high', // eslint-disable-line camelcase
            });

            test('sets online property to false on OFFLINE event', () => {
                backend._events.emit(TwitchEvents.OFFLINE);
                expect(store.dispatch).toHaveBeenCalledWith({
                    type: ACTION_SET_ONLINE,
                    online: false,
                });
            });
        });

        describe('loading spinner', () => {
            showLoadingEvents.forEach(evt => {
                test(`${evt} shows loading spinner`, () => {
                    backend._events.emit(evt, {});
                    expect(store.dispatch).toHaveBeenCalledWith({
                        type: ACTION_SET_LOADING,
                        isLoading: true,
                    });
                });
            });

            hideLoadingEvents.forEach(evt => {
                test(`${evt} hides loading spinner`, () => {
                    backend._events.emit(evt, {});
                    expect(store.dispatch).toHaveBeenCalledWith({
                        type: ACTION_SET_LOADING,
                        isLoading: false,
                    });
                });
            });
        });

        describe('quality change event', () => {
            const triggerQualityEvent = ({ quality, variant, isAuto }) => {
                store.dispatch(selectQuality(quality));
                backend._events.emit(TwitchEvents.QUALITY_CHANGE, {
                    quality: variant,
                    isAuto,
                });
                return store.getState().quality.available;
            };

            test('should set current quality in store', () => {
                const variant = '480p30';
                const isAuto = true;
                triggerQualityEvent({
                    quality: 'auto',
                    variant,
                    isAuto,
                });
                expect(store.dispatch).toHaveBeenCalledWith(setCurrentQuality(variant, isAuto));
            });

            test('should know if quality changes via auto or not', () => {
                const variant = '480p30';
                triggerQualityEvent({
                    quality: 'auto',
                    variant,
                    isAuto: true,
                });
                expect(store.dispatch).toHaveBeenCalledWith(setCurrentQuality(variant, true));
                store.dispatch.mockReset();

                triggerQualityEvent({
                    quality: 'auto',
                    variant,
                    isAuto: false,
                });

                expect(store.dispatch).toHaveBeenCalledWith(setCurrentQuality(variant, false));
            });

            test('should not update qualities if variant doesn\'t exist in qualities', () => {
                const quality = 'invalid';
                const variant = 'also invalid';

                store.dispatch(selectQuality(quality));
                backend._events.emit(TwitchEvents.QUALITY_CHANGE, {
                    quality: variant,
                    isAuto: false,
                });

                const actual = store.getState().quality.available;
                const expected = createMockQualities();
                expect(actual).toEqual(expected);
            });

            test('should update qualities when ABS_STREAM_FORMAT_CHANGE event is fired', () => {
                const callbackSpy = jest.fn();
                video.addEventListener(TwitchEvents.ABS_STREAM_FORMAT_CHANGE, callbackSpy);

                triggerQualityEvent({
                    quality: 'auto',
                    variant: 'medium',
                    isAuto: true,
                });
                const payload = {
                    stream_format_current: 'medium', // eslint-disable-line camelcase
                    stream_format_previous: 'low2', // eslint-disable-line camelcase
                };

                backend._events.emit(TwitchEvents.ABS_STREAM_FORMAT_CHANGE, payload);
                const [result] = callbackSpy.mock.calls[0];
                expect(result).toEqual(payload);
                const actual = store.getState().quality.available;
                const expected = createMockQualities();
                expect(actual).toEqual(expected);
            });
        });

        describe('playback management', () => {
            test('ACTION_PAUSE is called on pause', () => {
                const action = {
                    type: ACTION_PAUSE,
                };

                backend._events.emit(MediaEvents.PAUSE);
                expect(store.dispatch).toHaveBeenCalledWith(action);
            });

            test('ACTION_WAITING is called on waiting', () => {
                const action = {
                    type: ACTION_WAITING,
                };

                backend._events.emit(MediaEvents.WAITING);
                expect(store.dispatch).toHaveBeenCalledWith(action);
            });

            // eslint-disable-next-line max-len
            test('ACTION_INCREMENT_BUFFER_EMPTIES is called on increment bufferEmpties count', () => {
                const action = {
                    type: ACTION_INCREMENT_BUFFER_EMPTIES,
                };

                backend._events.emit(MediaEvents.WAITING);
                expect(store.dispatch).toHaveBeenCalledWith(action);
            });

            test('ACTION_PLAYING is called on playing', () => {
                const action = {
                    type: ACTION_PLAYING,
                };

                backend._events.emit(MediaEvents.PLAYING);
                expect(store.dispatch).toHaveBeenCalledWith(action);
            });

            test('ACTION_ENDED is called on ended', () => {
                const action = {
                    type: ACTION_ENDED,
                };

                backend._events.emit(MediaEvents.ENDED);
                expect(store.dispatch).toHaveBeenCalledWith(action);
            });

            test('ACTION_PLAYER_SEEKED is called on seeked', () => {
                const action = {
                    type: ACTION_PLAYER_SEEKED,
                    currentTime: 0,
                };

                backend.setCurrentTime(0);
                backend._events.emit(MediaEvents.SEEKED);
                expect(store.dispatch).toHaveBeenCalledWith(action);
            });
        });
        describe('public API methods', () => {
            let video;
            let backend;
            let store;
            const expectedBroadcastId = 12345;
            const expectedPlaySessionId = 67890;
            const expectedStats = {
                stat1: 'a stat',
            };

            beforeEach(() => {
                store = initStore({
                    stream: {
                        contentType: CONTENT_MODE_LIVE,
                    },
                    analytics: {
                        playSessionId: expectedPlaySessionId,
                    },
                    manifestInfo: {
                        // eslint-disable-next-line camelcase
                        broadcast_id: expectedBroadcastId,
                    },
                    stats: {
                        videoStats: expectedStats,
                    },
                });

                video = new Video(rootElm, store, {
                    backend: 'mediaplayer',
                });
                backend = store.getState().backend;
            });

            afterEach(() => {
                video.destroy();
                backend.destroy();
                store = null;
            });

            describe('that return store state', () => {
                test('getSessionInfo', () => {
                    expect(video.getSessionInfo()).toEqual({
                        broadcastId: expectedBroadcastId,
                        playSessionId: expectedPlaySessionId,
                    });
                });

                test('getPlaybackStats', () => {
                    expect(video.getPlaybackStats()).toEqual(expectedStats);
                });

                test('getStats', () => {
                    expect(video.getStats()).toEqual(expectedStats);
                });
            });

            describe('store subscriptions', () => {
                test('settings.persistenceEnabled', () => {
                    const action = {
                        type: ACTION_SET_PERSISTENCE,
                        value: false,
                    };

                    const toggleListener = jest.fn();
                    video.addEventListener(TwitchEvents.PERSISTENT_PLAYER_TOGGLE, toggleListener);

                    store.dispatch(action);

                    expect(toggleListener).toHaveBeenCalledTimes(1);
                });
            });
        });
    });

    describe('backend selection', () => {
        let store;
        let fakeTracker;

        beforeEach(() => {
            store = initStore({
                stream: {
                    contentType: CONTENT_MODE_LIVE,
                    restrictedBitrates: [],
                },
            });

            fakeTracker = { trackEvent: jest.fn() };
            store.dispatch(setAnalyticsTracker(fakeTracker));
            jest.spyOn(store, 'dispatch');
        });

        afterEach(() => {
            store = null;
        });

        describe('when no backends are available', () => {
            let video;

            beforeEach(() => {
                BackendUtils.getAvailableBackends = jest.fn().mockReturnValue([]);

                video = new Video(rootElm, store, {
                    backend: 'mediaplayer',
                });
            });

            afterEach(() => {
                BackendUtils.getAvailableBackends.mockRestore();
                video.destroy();
            });

            test('should fire a PLAYBACK_ERROR event', () => {
                expect(fakeTracker.trackEvent).toHaveBeenCalledTimes(1);

                const [eventName, eventProperties] = fakeTracker.trackEvent.mock.calls[0];

                expect(eventName).toBe(PLAYBACK_ERROR);
                expect(eventProperties).toEqual({
                    // eslint-disable-next-line camelcase
                    playback_error_code: 6000,
                    // eslint-disable-next-line camelcase
                    playback_error_msg: 'no_backend_supported',
                });
            });

            test('should set a renderer not available error code in store', () => {
                expect(store.dispatch).toHaveBeenCalledWith({
                    type: 'set error',
                    code: Errors.CODES.RENDERER_NOT_AVAILABLE,
                });
            });

            test('should load a blank backend', () => {
                expect(video.getBackend()).toBe(BACKEND_BLANK_TYPE);
            });
        });

        describe('when backends are available', () => {
            beforeEach(() => {
                BackendUtils.getAvailableBackends = jest.fn().mockReturnValue([
                    BACKEND_PLAYER_CORE,
                    BACKEND_FLASH,
                    BACKEND_MEDIA_PLAYER,
                ]);
            });

            afterEach(() => {
                BackendUtils.getAvailableBackends.mockRestore();
            });
            test('if requested backend is available, load it', () => {
                const video = new Video(rootElm, store, {
                    backend: BACKEND_FLASH,
                });

                expect(video.getBackend()).toBe(BACKEND_FLASH);
                video.destroy();
            });

            test('if requested backend is not available, load first backend in array', () => {
                const video = new Video(rootElm, store, {
                    backend: BACKEND_HLS,
                });

                expect(video.getBackend()).toBe(BACKEND_PLAYER_CORE);
                video.destroy();
            });

            test('if requested backend is not specified, load first backend in array', () => {
                const video = new Video(rootElm, store, {});
                expect(video.getBackend()).toBe(BACKEND_PLAYER_CORE);
                video.destroy();
            });
        });
    });
});
