/* global __dirname, process, module */
const path = require('path');
const webpack = require('webpack');
const merge = require('lodash/merge');

const HtmlWebpackPlugin = require('html-webpack-plugin');
const ExtractTextPlugin = require('extract-text-webpack-plugin');
const CopyWebpackPlugin = require('copy-webpack-plugin');
const SymlinkPlugin = require('./util/webpack-symlink-plugin.js');
const ConditionalPlugin = require('./util/webpack-conditional-plugin.js');

const ROOT_PATH = path.resolve(__dirname, '..');
const DEST_PATH = path.resolve(__dirname, '..', 'build');
const SRC_PATH = path.resolve(__dirname, '..', 'src');

const DEV_MODE = (process.env.NODE_ENV !== 'production');
const SOURCE_MAP = (DEV_MODE ? 'source-map' : false);

const CURRENT_DATE = new Date().toISOString().
  slice(0, -5).
  replace(/:/g, '').
  replace(/-/g, '.').
  replace('T', '-');
const CURRENT_COMMIT_HASH = process.env.GIT_COMMIT || 'DEV';
const APP_VERSION = `${CURRENT_DATE}+${CURRENT_COMMIT_HASH}`;

const baseOptions = require('./webpack-base.js');

module.exports = merge({}, baseOptions, {
    devtool: SOURCE_MAP,
    entry: {
        player: path.resolve(SRC_PATH, 'js', 'player'),
    },
    node: {
        fs: 'empty',
    },
    output: {
        path: DEST_PATH,
        filename: DEV_MODE ? 'js/[name].js' : 'js/[name].[hash].js',
        libraryTarget: 'umd',
        library: ['Twitch', 'video'],
        sourceMapFilename: '[file].map',
    },
    plugins: [
        new ExtractTextPlugin({
            filename: DEV_MODE ? 'css/[name].css' : 'css/[name].[contenthash].css',
        }),
        new ConditionalPlugin(!DEV_MODE, new webpack.optimize.UglifyJsPlugin()),
        new HtmlWebpackPlugin({
            filename: DEV_MODE ? 'index.html' : `index.${Date.now()}.html`,
            template: path.resolve(SRC_PATH, 'html', 'index.html'),
            favicon: path.resolve(ROOT_PATH, 'img', 'favicon.png'),
            minify: {
                collapseWhitespace: !DEV_MODE,
                minifyJS: !DEV_MODE,
                removeRedundantAttributes: true,
                removeComments: !DEV_MODE,
                useShortDoctype: true,
            },
        }),
        new SymlinkPlugin(),
        new CopyWebpackPlugin([{
            from: 'localization',
            to: 'localization',
        }], { ignore: ['*.pot', '*.po'] }),
        new webpack.DefinePlugin({
            'process.env.NODE_ENV': JSON.stringify(process.env.NODE_ENV || 'development'),
            APP_VERSION: JSON.stringify(APP_VERSION),
            I18N_LOCALES: JSON.stringify(require('./smartling.config.json').locales),
        }),
        new ConditionalPlugin(!DEV_MODE, new webpack.optimize.ModuleConcatenationPlugin()),
    ],
});
