job {
    name "video-player-ui-build"
    using 'TEMPLATE-autobuild'

    logRotator {
        daysToKeep(14)
    }

    wrappers {
        sshAgent 'git-aws-read-key'
        preBuildCleanup()
        timestamps()
        credentialsBinding {
            string('GITHUB_ACCESS_TOKEN', 'devtools-deployment-github-api-token')
            file('COURIERD_PRIVATE_KEY', 'courierd')
            file('AWS_CONFIG_FILE', 'aws_config')
        }
    }

    scm {
        git {
            remote {
                github 'video/player-ui', 'ssh', 'git-aws.internal.justin.tv'
                credentials 'git-aws-read-key'
            }
            clean true
        }
    }

    // If manta fails, don't deploy anything, otherwise deploy to the correct env.
    steps {
        shell 'rm -rf .manta'
        shell 'manta -v -proxy'
        saveDeployArtifact 'video/player-ui', '.manta'
    }

    throttleConcurrentBuilds {
        maxTotal 4
        maxPerNode 1
    }

    publishers {
        githubCommitNotifier()
        flexiblePublish {
            conditionalAction {
                condition {
                    status('SUCCESS', 'SUCCESS')
                }
                steps {
                    shell """
                        export AUTO_DEPLOY_ENV=\$(./scripts/auto-deploy-env.sh \$GIT_BRANCH)
                        export BRANCH=\$(echo "\$GIT_BRANCH" | sed 's/origin\\///')
                        ./scripts/auto-deploy.py -b "\$BRANCH" -e "\$AUTO_DEPLOY_ENV" -c "\$GIT_COMMIT" -t "\$GITHUB_ACCESS_TOKEN"
                    """
                }
            }
        }
    }
}

job {
    name "TEMPLATE-video-player-ui-deploy"
    using "TEMPLATE-deploy"

    logRotator {
        daysToKeep(14)
    }

    parameters {
        stringParam('BRANCH', '', 'branch name being deployed')
    }

    wrappers {
        credentialsBinding {
            file('AWS_CONFIG_FILE', 'aws_config')
            string 'PLAYER_AWS_ACCESS_KEY', 'video-player-assets-access-key'
            string 'PLAYER_AWS_SECRET_KEY', 'video-player-assets-secret-key'
        }
    }

    steps {
        downloadDeployArtifact 'video/player-ui'
        shell """
            export AWS_ACCESS_KEY_ID=\$PLAYER_AWS_ACCESS_KEY
            export AWS_SECRET_ACCESS_KEY=\$PLAYER_AWS_SECRET_KEY
            deploy/upload.sh -b "\$BRANCH" -d dist -e "\$ENVIRONMENT"
            echo 'UPLOAD COMPLETE'
        """
    }
}

// Clean-deploy will fire this job
job {
    name 'video-player-ui-deploy-production'
    using 'TEMPLATE-video-player-ui-deploy'
}

// Clean-deploy will fire this job
job {
    name 'video-player-ui-deploy-beta'
    using 'TEMPLATE-video-player-ui-deploy'

    publishers {
        flexiblePublish {
            conditionalAction {
                condition {
                    status('SUCCESS', 'SUCCESS')
                }
                steps {
                    downstreamParameterized {
                        trigger('video-player-ui-smoca-tests') {
                            parameters {
                                currentBuild()
                                predefinedProp('PLAYER_ENVIRONMENT', 'betaplayer')
                            }
                        }
                    }
                }
            }
        }
    }
}

// Clean-deploy will fire this job
job {
    name 'video-player-ui-deploy-test'
    using 'TEMPLATE-video-player-ui-deploy'

    publishers {
        flexiblePublish {
            conditionalAction {
                condition {
                    status('SUCCESS', 'SUCCESS')
                }
                steps {
                    downstreamParameterized {
                        trigger('video-player-ui-smoca-tests') {
                            parameters {
                                currentBuild()
                                predefinedProp('PLAYER_ENVIRONMENT', 'testplayer')
                            }
                        }
                    }
                }
            }
        }
    }
}

job {
    name 'video-player-ui-smoca-tests'
    concurrentBuild true

    parameters {
      stringParam('BRANCH', '', 'Branch to be tested')
      stringParam('PLAYER_ENVIRONMENT', '', 'Testing Environment')
      stringParam('GIT_COMMIT', '', 'git commit')
    }

    steps {
        downstreamParameterized {
            // Beta test for Windows 10, Chrome - Trigger a smoca-player-ui-win10 job against the branch
            // Not doing this in smoca-player-ui-suite so that failures are not reported to the PR
            trigger('smoca-player-ui-win10', "ALWAYS") {
                parameters {
                    currentBuild()
                    predefinedProp('BROWSER', 'Chrome')
                    predefinedProp('OS_PLATFORM', 'Win10')
                    predefinedProp('PLAYER_BRANCH', '$BRANCH')
                    predefinedProp('PLAYER_ENVIRONMENT', '$PLAYER_ENVIRONMENT')
                    predefinedProp('PLAYER_TYPE', 'mediaplayer')
                }
            }

            // Beta test for Windows 10, Firefox - Trigger a smoca-player-ui-win10 job against the branch
            // Not doing this in smoca-player-ui-suite so that failures are not reported to the PR
            trigger('smoca-player-ui-win10', "ALWAYS") {
                parameters {
                    currentBuild()
                    predefinedProp('BROWSER', 'Firefox')
                    predefinedProp('OS_PLATFORM', 'Win10')
                    predefinedProp('PLAYER_BRANCH', '$BRANCH')
                    predefinedProp('PLAYER_ENVIRONMENT', '$PLAYER_ENVIRONMENT')
                    predefinedProp('PLAYER_TYPE', 'mediaplayer')
                }
            }

            trigger('smoca-player-ui-suite', "ALWAYS") {
                block {
                    buildStepFailure('FAILURE')
                }
                parameters {
                    currentBuild()
                    predefinedProp('BROWSER', 'chrome')
                    predefinedProp('PLAYER_BRANCH', '$BRANCH')
                    predefinedProp('PLAYER_GIT_COMMIT', '$GIT_COMMIT')
                    predefinedProp('PLAYER_CHANNEL', 'monstercat')
                }
            }
            trigger('smoca-player-ui-suite', "ALWAYS") {
                block {
                    buildStepFailure('FAILURE')
                }
                parameters {
                    currentBuild()
                    predefinedProp('BROWSER', 'firefox')
                    predefinedProp('PLAYER_BRANCH', '$BRANCH')
                    predefinedProp('PLAYER_GIT_COMMIT', '$GIT_COMMIT')
                    predefinedProp('PLAYER_CHANNEL', 'monstercat')
                }
            }
        }
    }
    publishers {
        flexiblePublish {
            conditionalAction {
                condition {
                    and {
                        status('SUCCESS', 'SUCCESS')
                    } {
                        stringsMatch('$PLAYER_ENVIRONMENT', 'betaplayer', false)
                    }
                }
                steps {
                    downstreamParameterized {
                        trigger('video-player-ui-continuous-deploy') {
                            parameters {
                                currentBuild()
                            }
                        }
                    }
                }
            }
        }
    }
}

job {
    name "video-player-ui-continuous-deploy"
    parameters {
        stringParam('GIT_COMMIT', '', 'git commit')
    }
    wrappers {
        sshAgent 'git-aws-read-key'
        preBuildCleanup()
        timestamps()
        credentialsBinding {
            string('GITHUB_ACCESS_TOKEN', 'devtools-deployment-github-api-token')
        }
    }
    scm {
        git {
            remote {
                github 'video/player-ui', 'ssh', 'git-aws.internal.justin.tv'
                credentials 'git-aws-read-key'
            }
            clean true
            branches '${GIT_COMMIT}'
        }
    }

    steps {
        shell """
            echo "\$GIT_COMMIT"
            echo "\$GITHUB_ACCESS_TOKEN"
            ./scripts/auto-deploy.py -b master -e production -c \$GIT_COMMIT -t \$GITHUB_ACCESS_TOKEN
        """
    }
}

job {
    name "video-player-ui-branch-builder"

    parameters {
        stringParam('VIDEO_REPO', 'player-ui', 'Repo name')
    }

    scm {
        git {
            remote {
                github 'video/player-ui', 'ssh', 'git-aws.internal.justin.tv'
                credentials 'git-aws-read-key'
            }
            clean true
        }
    }

    concurrentBuild true

    throttleConcurrentBuilds {
        maxTotal 4
        maxPerNode 1
    }

    logRotator {
        daysToKeep(14)
    }

    triggers {
        githubPush()
    }

    wrappers {
        sshAgent 'git-aws-read-key'
        preBuildCleanup()
        timestamps()
        credentialsBinding {
            string('GITHUB_ACCESS_TOKEN', 'devtools-deployment-github-api-token')
        }
        buildName('#\${BUILD_NUMBER} (\${GIT_BRANCH})')
    }


    steps {
        shell """./scripts/setup-e2e-env.sh"""

        // Kick off web-stage env creator job
        downstreamParameterized {
            trigger('web-web-stage_env_creator') {
                block {
                    buildStepFailure('FAILURE')
                    failure('FAILURE')
                    unstable('UNSTABLE')
                }
                parameters {
                    propertiesFile("web/ENV_CREATOR.properties", true)
                }
            }
        }

        // Wait for video-${VIDEO_REPO} deployment (kicked off by video-${VIDEO_REPO}-deploy-test job)
        shell """./scripts/poll-e2e-env-readiness.sh"""
    }
}

job {
    name "video-player-ui-builder-cleanup"

    triggers {
        cron('H/30 * * * *')
    }

    logRotator {
        daysToKeep(14)
    }

    wrappers {
        timestamps()
        sshAgent 'git-aws-read-key'
    }

    parameters {
        stringParam('VIDEO_REPO', 'player-ui', 'Repo name')
    }

    steps {
        shell """./scripts/destroy-e2e-env-resources.sh"""

        // Kick off web-stage env creator job
        downstreamParameterized {
            trigger('web-web-stage_env_creator') {
                block {
                    buildStepFailure('FAILURE')
                    failure('FAILURE')
                    unstable('UNSTABLE')
                }
                parameters {
                    propertiesFile("PURGE_PARAMS.properties", true)
                }
            }
        }
    }
}
// Smoca

job {
  name 'smoca-channel-integration'
  concurrentBuild()

  throttleConcurrentBuilds {
    maxTotal(16)
  }

  label 'qaslave'

  parameters {
    stringParam('BROWSER', 'firefox', "Browser to run tests against.")
    stringParam('BROWSER_VERSION', '', "Browser version to run tests against.")
    stringParam('OS_PLATFORM', 'Windows', "OS to run tests against.")
    stringParam('OS_VERSION', '10', "BETA: OS Version to run tests against.")
    choiceParam('GRID_HOST', ['internal', 'saucelabs', 'internal_dev'])
    choiceParam('LOCAL_TESTING', ['false', 'true'], 'Run tests with Local Testing.')
  }

  scm {
    git {
      remote {
        github 'video/player-ui', 'ssh', 'git-aws.internal.justin.tv'
        credentials 'git-aws-read-key'
      }
      branches 'origin/master'
      clean true
    }
  }

  wrappers {
    credentialsBinding {
      string 'DROPBOX_TOKEN', 'smoca-dropbox-token'
      usernamePassword('SAUCELABS_USER', 'SAUCELABS_KEY', 'saucelabs-smoca-video')
    }
    sshAgent 'git-aws-read-key'
    colorizeOutput(colorMap = 'xterm')
    timestamps()
  }

  steps {
    shell """
      |unset HTTP_PROXY
      |unset http_proxy
      |unset HTTPS_PROXY
      |unset https_proxy
      |unset NO_PROXY
      |unset no_proxy
      |cat <<EOF>smoca/buildenv.sh
      |export RETRY_COUNT=5
      |export GRID=true
      |export GRID_HOST=\$GRID_HOST
      |export BROWSER_NAME="\$BROWSER"
      |export BROWSER_VERSION=\$BROWSER_VERSION
      |export OS_PLATFORM=\$OS_PLATFORM
      |export OS_VERSION=\$OS_VERSION
      |export DROPBOX_TOKEN=\$DROPBOX_TOKEN
      |export SCREENSHOTS=true
      |export SCREEN_RESOLUTION='1600x1200'
      |export SAUCELABS_USERNAME=\$SAUCELABS_USER
      |export SAUCELABS_ACCESS_KEY=\$SAUCELABS_KEY
      |export LOCAL_TESTING=\$LOCAL_TESTING
      |
      |EOF
      |cd smoca
      |set +x
      |manta -f smoca_infra/.manta.json
      |set -x
    """.stripMargin()
  }
}

multiJob('smoca-channel-integration-suite') {
  concurrentBuild()
  throttleConcurrentBuilds {
    maxTotal(4)
  }

  parameters {
    stringParam('PLAYER_GIT_COMMIT', '', 'Commit to test')
  }

  scm {
    git {
      remote {
        github 'video/player-ui', 'ssh', 'git-aws.internal.justin.tv'
        credentials 'git-aws-read-key'
      }
      branches 'origin/master'
      clean true
    }
  }

  wrappers {
    sshAgent 'git-aws-read-key'
    colorizeOutput(colorMap = 'xterm')
    timestamps()
    credentialsBinding {
      string('GITHUB_ACCESS_TOKEN', 'devtools-deployment-github-api-token')
    }
  }

  steps {
    phase('Channel Page') {
      phaseJob('smoca-channel-integration') {
        killPhaseCondition('NEVER')
        parameters {
          currentBuild()
          predefinedProp('BROWSER', 'chrome')
          predefinedProp('BROWSER_VERSION', '59')
          predefinedProp('OS_PLATFORM', 'Windows')
          predefinedProp('OS_VERSION', '10')
          predefinedProp('GRID_HOST', 'saucelabs')
          predefinedProp('LOCAL_TESTING', 'true')
        }
      }

      phaseJob('smoca-channel-integration') {
        killPhaseCondition('NEVER')
        parameters {
          currentBuild()
          predefinedProp('BROWSER', 'Edge')
          predefinedProp('BROWSER_VERSION', '14')
          predefinedProp('OS_PLATFORM', 'Windows')
          predefinedProp('OS_VERSION', '10')
          predefinedProp('GRID_HOST', 'saucelabs')
          predefinedProp('LOCAL_TESTING', 'true')
        }
      }

      phaseJob('smoca-channel-integration') {
        killPhaseCondition('NEVER')
        parameters {
          currentBuild()
          predefinedProp('BROWSER', 'internet explorer')
          predefinedProp('BROWSER_VERSION', '11.103')
          predefinedProp('OS_PLATFORM', 'Windows')
          predefinedProp('OS_VERSION', '10')
          predefinedProp('GRID_HOST', 'saucelabs')
          predefinedProp('LOCAL_TESTING', 'true')
        }
      }

      phaseJob('smoca-channel-integration') {
        killPhaseCondition('NEVER')
        parameters {
          currentBuild()
          predefinedProp('BROWSER', 'firefox')
          predefinedProp('BROWSER_VERSION', '')
          predefinedProp('OS_PLATFORM', 'Win8')
          predefinedProp('OS_VERSION', '')
          predefinedProp('GRID_HOST', 'prod')
          predefinedProp('LOCAL_TESTING', 'true')
        }
      }
    }
  }
}
