#!/usr/bin/env python

import re
import sys
import logging
import argparse
import requests
import subprocess

CLEAN_DEPLOY_HOST = 'https://clean-deploy.internal.justin.tv'
GIT_HOST = 'https://git-aws.internal.justin.tv'


def main():
    try:
        options = parse_options()
        log = logging.getLogger('AutoDeploy')

        if options.verbose:
            logging.basicConfig(level=logging.DEBUG)
        else:
            logging.basicConfig(level=logging.INFO)

        for key, value in vars(options).iteritems():
            if value is None:
                log.error('Aborting auto-deploy: Argument \'{0}\' is required'.format(key))
                return 1

        sha1_regex_pattern = re.compile(r'\b[0-9a-f]{40}\b')

        if not sha1_regex_pattern.match(options.token):
            log.error('Aborting auto-deploy: token option is not valid sha-1 format')
            return 1

        if not sha1_regex_pattern.match(options.commit):
            log.error('Aborting auto-deploy: commit option is not valid sha-1 format')
            return 1

        (is_frozen, reason) = get_freeze_status(options.environment)

        if is_frozen:
            log.error('Aborting auto-deploy. The {0} environment is frozen. Reason: {1}'.format(options.environment, reason))
            return 1

        trigger_deploy(options.commit, options.branch, options.environment, options.token)
        return 0
    except requests.exceptions.RequestException as e:
        log.exception('Aborting auto-deploy: {0}'.format(e.response.text))
    except Exception as e:
        log.exception('Aborting auto-deploy: {0}'.format(e))
        return 1


def parse_options():
    parser = argparse.ArgumentParser()
    parser.add_argument('-b', '--branch', help='branch to deploy')
    parser.add_argument('-e', '--environment', help='environment to deploy to')
    parser.add_argument('-c', '--commit', help='commit to deploy')
    parser.add_argument('-t', '--token', help='github access token')
    parser.add_argument('-v', '--verbose', help='enable debug output', action='store_true')

    return parser.parse_args()


def get_freeze_status(environment):
    log = logging.getLogger('AutoDeploy')
    log.info('Fetching {0} environment freeze status'.format(environment))

    url = '{0}/v1/freeze/list'.format(CLEAN_DEPLOY_HOST)
    params = {'owner': 'video', 'repo': 'player-ui'}
    response = requests.get(url, params=params)
    response.raise_for_status()

    freeze_list = response.json()

    for value in freeze_list:
        if value['is_frozen'] and value['environment'] == environment:
            return (True, value['reason'])

    return (False, '')


def trigger_deploy(commit, branch, environment, token):
    log = logging.getLogger('AutoDeploy')
    log.info('Triggering artifact deployment')

    description = subprocess.check_output('git show {0} --format=\'%s\' | head -1'.format(commit), shell=True)

    url = '{0}/v1/deployments'.format(CLEAN_DEPLOY_HOST)
    headers = {'GithubAccessToken': token}
    payload = {
        'description': description,
        'owner': 'video',
        'repo': 'player-ui',
        'ref': commit,
        'branch': branch,
        'environment': environment,
        'codeReviewURL': get_code_review_url(commit, environment),
        'severity': 'auto-deploy,'
    }

    response = requests.post(url, headers=headers, data=payload)
    response.raise_for_status()

    log.info('Successfully auto-deployed {0} branch to {1} environment.'.format(branch, environment))


def get_code_review_url(commit, environment):
    log = logging.getLogger('AutoDeploy')
    last_deploy_data = {}
    try:
        url = '{0}/v1/artifact/compare'.format(CLEAN_DEPLOY_HOST)
        params = {'env': environment, 'name': 'player-ui', 'owner': 'video', 'rev': 'master'}
        response = requests.get(url, params=params)
        last_deploy_data = response.json()
    except (request.exceptions.RequestException, ValueError):
        pass

    if 'previous_sha' in last_deploy_data:
        return '{0}/video/player-ui/compare/{1}...{2}'.format(GIT_HOST, last_deploy_data['previous_sha'], commit)
    else:
        log.warning('Can not find previously deployed SHA for {0}. Code review URL will only include the deploying commit.'.format(environment))
        return '{0}/video/player-ui/commit/{1}'.format(GIT_HOST, commit)


if __name__ == "__main__":
    sys.exit(main())
