#!/usr/bin/env bash

BASELINE=BASELINE_LIST.txt
CURRENT=CURRENT_LIST.txt
TEMP_DIFF=TEMP_DIFF.tmp
# test
# Update the baseline list of branches
if [ -f ${CURRENT} ]; then
  mv ${CURRENT} ${BASELINE}
  echo "Updated baseline list of branches"
fi

# Generate a list of current branches
git ls-remote git@git-aws.internal.justin.tv:video/player-ui.git | grep  refs/heads | cut -d/ -f3- | sort > ${CURRENT}
# TODO: add player-flash

# Is there a baseline? Only generate diff (and create a purge queue) if there's a baseline
if [ -f ${BASELINE} ]; then
  echo ""
  echo "Found baseline list of branches:"
  cat ${BASELINE}
  echo ""
  echo "Current list of branches:"
  cat ${CURRENT}

  # Cache branch owners, because we cannot retrieve branch owner name after a branch is deleted
  touch BRANCH_OWNER_CACHE.csv
  while IFS= read -r BRANCH; do
    # shellcheck disable=SC2094
    if ! grep "${BRANCH}" BRANCH_OWNER_CACHE.csv; then
      BRANCH_OWNER=$(curl -s -H "Authorization: token ${GITHUB_ACCESS_TOKEN}" https://git-aws.internal.justin.tv/api/v3/repos/video/player-ui/branches/"${BRANCH}" | jq ".commit.author.login" | tr -d \")
      echo "${BRANCH}:${BRANCH_OWNER}" >> BRANCH_OWNER_CACHE.csv
    fi
  done < BRANCH_OWNER_CACHE.csv
  # Keep only the last 200 branch name, owner info
  tail -200 BRANCH_OWNER_CACHE.csv > BRANCH_OWNER_CACHE.csv.tmp
  mv BRANCH_OWNER_CACHE.csv.tmp BRANCH_OWNER_CACHE.csv

  diff -du ${BASELINE} ${CURRENT} > ${TEMP_DIFF}
  if [ -s ${TEMP_DIFF} ]; then
    # Find the diff that contains ONLY the removed branches (indicated by ^- in diff)
    tail -n +4 ${TEMP_DIFF} | grep ^- | cut -c2- > PURGE_QUEUE.txt
    if [ -s PURGE_QUEUE.txt ]; then
      echo ""
      echo "Purge queue:"
      cat PURGE_QUEUE.txt
      echo ""
      # To be safe, do not proceed with environment purging if purge queue is longer than 3 branches.
      # It may indicate some bug in script or unusual process.
      # User must run this job with FORCE_DELETE_LONG_LIST_OF_ENVS set to true to unblock
      # shellcheck disable=SC2002
      if [ "$(cat PURGE_QUEUE.txt | wc -l)" -gt 3 ]; then
        echo "WARNING! There were more than 3 deleted branches. There could be a script or abnormal process that occurred. Check the following list:"
        if [ "${FORCE_DELETE_LONG_LIST_OF_ENVS}" != "true" ]; then
          echo "FORCE_DELETE_LONG_LIST_OF_ENVS is not set to true. Will not purge environments unless human intervention overrides this safety check by manually running this job and setting FORCE_DELETE_LONG_LIST_OF_ENVS to true"
          rm -f PURGE_QUEUE.txt
          exit 1
        fi
      fi
    else
      echo ""
      echo "New branches found. However, no deleted branches found at this time."
    fi
  else
    echo ""
    echo "No deleted branches found at this time."
  fi
else
  echo ""
  echo "Baseline list of branches not known. Will not purge environments this round."
fi


# Clean up temp files
rm -f PURGE_PARAMS.properties*
rm -f ${TEMP_DIFF}

# Check the purge queue. If not empty, pop ONE entry and destroy that environment.
# DO NOT PURGE multiple environments in one go as it is too risky.
if [ -s PURGE_QUEUE.txt ]; then

  # Get the shortened TFCI ENV name
  BRANCH_NAME=$(head -1 PURGE_QUEUE.txt)

  if [ "${BRANCH_NAME}" == "" ]; then
    # If branch name is empty *somehow*, do not proceed with automation
    # One possibility is someone/something triggered workspace directory to be cleaned
    echo ""
    echo "Branch name to purge is somehow empty. Something bad happened in the workspace. Exiting."
    exit 1
  fi

  # Pop the queue
  sed -i 1d PURGE_QUEUE.txt
  TFCI_ENV=$(echo "${BRANCH_NAME}" | sed -e 's/_/-/g' -e 's/[^a-zA-Z0-9]//g' -e 's/^staging-\?//g' | cut -c1-19)

  # Put the remaining branch name(s) back in the current list, to consider them for deletion the next round.
  cat ${CURRENT} PURGE_QUEUE.txt > RESTORE.txt
  sort RESTORE.txt > ${CURRENT}
  rm -f RESTORE.txt

  # Create properties file for env_creator jenkins job (to delete web-stage)
  BRANCH_OWNER=$(grep -F "${BRANCH_NAME}:" BRANCH_OWNER_CACHE.csv | cut -d: -f2)
  echo "Will purge test environment for branch ${BRANCH_NAME}"
  echo "tfci_email=${BRANCH_OWNER}@justin.tv
tfci_env=${TFCI_ENV}
update_env=true
destroy_env=true
really_destroy_env=true" > PURGE_PARAMS.properties

  # DEBUG
  echo ""
  echo "Parameters for web-web-stage_env_creator:"
  cat PURGE_PARAMS.properties

  # TODO: call whatever tooling exists to delete the player-ui branch env

  # delete the temporary branch from web/web git repo
  echo ""
  BRANCH_DELETE_OUTPUT=$(curl --retry 3 -s -X DELETE -H "Authorization: token ${GITHUB_ACCESS_TOKEN}" https://git-aws.internal.justin.tv/api/v3/repos/web/web/git/refs/heads/"${BRANCH_NAME}")
  if [ "${BRANCH_DELETE_OUTPUT}" == "" ]; then
    echo "Successfully deleted temporary web/web branch ${BRANCH_NAME}"
  else
    echo "Failed to delete temporary web/web branch ${BRANCH_NAME}. Error:"
    echo "${BRANCH_DELETE_OUTPUT}"
  fi
fi


