#!/usr/bin/env bash

# Create temp branch and push to github
# REQUIRES ssh-agent running for git-aws-read-key
rm -rf web
git clone git@git-aws.internal.justin.tv:web/web.git
cd web || exit 1

COMMON_BRANCH="${GIT_BRANCH##origin/}"
if [ "${COMMON_BRANCH}" == "master" ]; then
  # Be paranoid about somehow triggering this on 'master' branch
  exit 0
fi

if git branch -r | grep "${COMMON_BRANCH}"; then
  echo "Branch already exists in origin. Reusing it"
else
  echo "Creating temporary branch ${COMMON_BRANCH} in web/web repo..."
  git branch "${COMMON_BRANCH}"
fi

git checkout "${COMMON_BRANCH}"
# Keep branch up to date with potentially moving origin/master
git merge origin/master -m "Merge current origin/master into branch"

if grep betaplayer.twitch.tv config/settings/staging.json; then
  sed -i "s/\\/\\/betaplayer.twitch.tv/\\/\\/testplayer.twitch.tv\\/${COMMON_BRANCH}/" ./config/settings/staging.json
  git diff
  git add ./config/settings/staging.json
  git commit -m "DO NOT MERGE TO MASTER. Point host.player to //testplayer.twitch.tv/${COMMON_BRANCH}. DO NOT MERGE TO MASTER"
else
  echo "The string 'betaplayer.twitch.tv' is no longer found in config/settings/staging.json"
  if grep testplayer.twitch.tv config/settings/staging.json; then
    echo "Video player is already pointing to a temporary test url. Automation will proceed."
  else
    echo "Video player is NOT already pointing to a temporary test url"
    echo "This is unexpected. Exiting automation immediately. Contact #qe- for troubleshooting."
    exit 1
  fi
fi

# At this point, proceed with automation: push local changes to origin to trigger web/web build
git status
git push --set-upstream origin "${COMMON_BRANCH}"

# Prepare build parameters for jenkins job: web-web-stage_env_creator
GIT_COMMITTER_EMAIL=$(curl --retry 3 -LSs -H "Authorization: token ${GITHUB_ACCESS_TOKEN}" https://git-aws.internal.justin.tv/api/v3/repos/video/player-ui/commits/"${GIT_COMMIT}" | jq ".commit.author.email" | tr -d '"')
echo GIT_COMMITTER_EMAIL="${GIT_COMMITTER_EMAIL}"
GITHUB_CREATOR=$(echo "${GIT_COMMITTER_EMAIL}" | cut -d@ -f1)

TFCI_ENV=$(echo "${COMMON_BRANCH}" | sed -e 's/_/-/g' -e 's/[^a-zA-Z0-9]//g' -e 's/^staging-\?//g' | cut -c1-19)
(echo tfci_email="${GIT_COMMITTER_EMAIL}"
echo tfci_env="${TFCI_ENV}"
echo update_env=true
echo tfci_meta=/video/player-ui/branch/"${COMMON_BRANCH}" ) > ENV_CREATOR.properties

echo "Parameters to web-web-stage_env_creator:"
cat ENV_CREATOR.properties

# Prepare build parameters for jenkins job: qa-smoca
echo "ENVIRONMENT=clean-staging-${TFCI_ENV}
GITHUB_CREATOR=${GITHUB_CREATOR}" > QA_SMOCA.properties
echo "Parameters to qa-smoca:"
cat QA_SMOCA.properties
