#!/usr/bin/env bash
#
# USAGE
#  ./test-embed.sh --video=v12345 --width=100 --height=200
#  ./test-embed.sh --channel=twitchmedia2
#
# OPTIONS
#   --height={HEIGHT} [default: 854]
#   --width={WIDTH} [default: 480]
#   --video=={VOD_ID} [default: off]
#   --channel={CHANNEL_NAME} [default: monstercat]
#   --keep-file [default: false]
#   --src [default: 'https://localhost.twitch.tv:8000/js/embed/v1.js']
#
# UPDATES
#   Please run shellcheck
#       - https://github.com/koalaman/shellcheck
#       - http://www.shellcheck.net/

# OPTIONS
CONTENT_ID="monstercat"
DELETE_FILE=true
HEIGHT=480
OUTPUT_FILENAME="test-embed.html"
PLAYER_SOURCE="https://localhost.twitch.tv:8000/js/embed/v1.js"
WIDTH=854

CLIENT_HOSTNAME="player.twitch.tv"
TEST_HOSTNAME="localhost.twitch.tv:8080"
TEST_PORT="$(echo $TEST_HOSTNAME | tr ':' '\n' | tail -1)"

CONTENT_TYPE="channel"
_CLIENT_FILE="../src/js/embed/client.js"

for i in "$@"; do
    case $i in
        --channel=*)
            CONTENT_TYPE="channel"
            CONTENT_ID="${i#*=}"
            shift
        ;;

        --height=*)
            HEIGHT="${i#*=}"
            shift
        ;;

        --keep-file)
            DELETE_FILE=false
            shift
        ;;

        --output=*)
            OUTPUT_FILENAME="${i#*=}"
            shift
        ;;

        --src=*)
            PLAYER_SOURCE=HEIGHT="${i#*=}"
            shift
        ;;

        --video=*)
            CONTENT_TYPE="video"
            CONTENT_ID="${i#*=}"

            # lowercase
            CONTENT_ID="${CONTENT_ID,,}"
            if [[ "${CONTENT_ID:0:1}" != "v" ]]; then
                CONTENT_ID="v${CONTENT_ID}"
            fi
            shift
        ;;

        --width=*)
            WIDTH="${i#*=}"
            shift
        ;;
    esac
done

cat > "$OUTPUT_FILENAME" <<- _EOF_
<!DOCTYPE html>
<html>
    <head>
        <title>Test Embed</title>
        <script src="${PLAYER_SOURCE}"></script>
    </head>
    <body>
        <div id="player"></div>
        <script type="text/javascript">
            var options = {
                "width": ${WIDTH},
                "height": ${HEIGHT},
                "${CONTENT_TYPE}": "${CONTENT_ID}"
            };

            window.player = new Twitch.Player("player", options);
        </script>
    </body>
</html>
_EOF_

# replace client.js
if [ -f "$_CLIENT_FILE" ]; then
    sed -i "" -e "s/${CLIENT_HOSTNAME}/${TEST_HOSTNAME}/" "$_CLIENT_FILE"
fi

# remove file on exit
trap '
    [ "$DELETE_FILE" = true ] && rm -f "$OUTPUT_FILENAME";
    [ -f "$_CLIENT_FILE" ] && sed -i "" -e "s/${TEST_HOSTNAME}/${CLIENT_HOSTNAME}/" "$_CLIENT_FILE";
    exit;
' INT TERM EXIT

# run mini-webserver
echo "Running embed-server at: http://${TEST_HOSTNAME}/${OUTPUT_FILENAME}"
while true; do
    { echo -ne "HTTP/1.1 200 OK\r\n\r\n"; cat "$OUTPUT_FILENAME"; } | nc -vl "$TEST_PORT"
done
