export default function transform(fileInfo, api) {
    const j = api.jscodeshift;
    const root = j(fileInfo.source);

    return chain(root, j)
        .then(assertDeepEqual)
        .then(assertOk)
        .then(assertNotOk)
        .then(assertEqual)
        .then(qunitModule)
        .then(qunitTest)
        .then(unitTest)
        .then(beforeEach)
        .then(afterEach)
        .then(expectAssertions)
        .toSource()
}

const qunitModule = createQUnitMod('module', 'describe');
const qunitTest = createQUnitMod('test', 'test');
const assertDeepEqual = createAssertTransform('deepEqual', 'toEqual');
const assertEqual = createAssertTransform('equal', 'toBe');
const assertOk = createAssertTransform('ok', 'toBeTruthy');
const assertNotOk = createAssertTransform('notOk', 'toBeFalsy');
const beforeEach = createHooksMod('beforeEach', 'beforeEach');
const afterEach = createHooksMod('afterEach', 'afterEach');

const unitTest = function(root, j) {
    return root
        .find(j.CallExpression, {
            callee: {
                type: "Identifier",
                name: 'unitTest',
            }
        })
        .replaceWith(nodePath => {
            const {
                node
            } = nodePath;
            node.callee.name = 'describe';
            return node;
        });
};

function expectAssertions(root, j) {
    return root
        .find(j.CallExpression, {
            callee: {
                type: "MemberExpression",
                object: {
                    type: 'Identifier',
                    name: 'assert',
                },
                property: {
                    type: 'Identifier',
                    name: 'expect',
                },
            },
        })
        .replaceWith(nodePath => {
            const {
                node
            } = nodePath;
            node.callee.object.name = 'expect'
            node.callee.property.name = 'assertions';
            return node;
        });
}

function createAssertTransform(method, jestMethod) {
    return function(root, j) {
        return root
            .find(j.CallExpression, {
                callee: {
                    type: "MemberExpression",
                    object: {
                        type: 'Identifier',
                        name: 'assert',
                    },
                    property: {
                        name: method,
                    }
                }
            })
            .replaceWith(nodePath => {
                const {
                    node
                } = nodePath;
                const args = node.arguments;
                node.callee = {
                    "type": "MemberExpression",
                    "object": {
                        "type": "CallExpression",
                        "callee": {
                            "type": "Identifier",
                            "name": "expect",
                        },
                        "arguments": [args[0]],
                    },
                    property: {
                        name: jestMethod,
                        type: 'Identifier'
                    },
                };
                node.arguments = isSingleParamAssert(method) ? [] : node.arguments.slice(1);
                return node;
            });
    }
}

function isSingleParamAssert(method) {
    return method === 'ok' ||
        method === 'notOk';
}

function createHooksMod(method, jestMethod) {
    return function(root, j) {
        return root.find(j.CallExpression, {
                callee: {
                    type: 'MemberExpression',
                    object: {
                        type: 'Identifier',
                        name: 'hooks',
                    },
                    property: {
                        type: 'Identifier',
                        name: method,
                    },
                },
            })
            .replaceWith(nodePath => {
                const {
                    node
                } = nodePath;
                node.callee.type = 'Identifier';
                node.callee.name = jestMethod;
                return node;
            });
    }
}

function createQUnitMod(method, jestMethod) {
    return function qunitTest(root, j) {
        return root.find(j.CallExpression, {
                callee: {
                    type: 'MemberExpression',
                    object: {
                        type: 'Identifier',
                        name: 'QUnit',
                    },
                    property: {
                        type: 'Identifier',
                        name: method,
                    },
                },
            })
            .replaceWith(nodePath => {
                const {
                    node
                } = nodePath;
                const args = node.arguments;
                args[1].params = [];
                node.callee.type = 'Identifier';
                node.callee.name = jestMethod;
                return node;
            });
    }
}

function chain(root, j) {
    return {
        then: function(cb) {
            cb(root, j);
            return chain(root, j)
        },
        toSource: function() {
            return root.toSource();
        }
    };
}
