import { AdsRequestContext } from '../ads/adscontext';
import { LiveTwitchContentStream } from 'stream/twitch-live';
import { VODTwitchContentStream } from 'stream/twitch-vod';
import { channelAPIInfo, channelInfo, channelViewerInfo,
         getCommunitiesFromChannel, userInfo, videoInfo, getChannelAdProperties } from '../api';
import { localStore } from 'util/storage';
import includes from 'lodash/includes';
import { LEVIATHAN, PADTIME, CHOMPERZ } from '../experiments';
import { parseUri } from '../util/parseuri';
import { KEY_MATURE } from '../ui/containers/overlays/mature-overlay';

export const DEFAULT_AD_DURATION = 30; // seconds

export const ACTION_SET_ADBLOCK_DETECTED = 'set adblock detected';
export const ACTION_SET_CLICKTHROUGH_URL = 'set clickthrough url';
export const ACTION_SET_AD_METADATA = 'set ad metadata';
export const ACTION_IMA_SCRIPT_LOADED = 'check to see if ima script loaded';
export const ACTION_SET_VOD_MIDROLL_ARCHIVE_SETTING = 'set vod midroll archive setting';

export const AdContentTypes = Object.freeze({
    NONE: 'none',
    IMA: 'ima',
    STITCHED: 'stitched',
});

export const AdRollTypes = Object.freeze({
    NONE: 'none',
    PREROLL: 'preroll',
    MIDROLL: 'midroll',
    POSTROLL: 'postroll',
});

// For ill-used ads action creators, return this empty action
// so that nothing is affected in statestore.
export const INVALID_AD_ACTION = {
    type: 'an invalid ad action',
};

const NULL_AD_METADATA = {
    contentType: AdContentTypes.NONE,
    rollType: AdRollTypes.NONE,
};

/**
 * Initiate a request for ads
 *
 * @param {String} rollType Type of ads to request (PREROLL, MIDROLL, POSTROLL)
 * @param {Number} duration Interval (in seconds) to fill with ads
 * @return {Action}
 */
export function requestAds(rollType, duration, forced = false, creativeId = 0, requestType = '') {
    return function(_, getState) {
        const { adsManager } = getState();

        return getRequestContext(rollType, duration, getState(), forced, creativeId, requestType).then(context => {
            // If this channel is mature, don't request ads until user accepts the mature overlay
            if (context.mature && !localStore.get(KEY_MATURE, false)) {
                return;
            }

            if (!getState().manifestInfo.suppress) {
                adsManager.requestAds(context);
            }
        });
    };
}

/**
 * Sets the current state of ads in the player
 *
 * @param {Object} {contentType: AdContentTypes, rollType: AdRollTypes}
 * @return {Action}
 */
export function setCurrentAdMetadata(adMetadata) {
    if (!isValidAdMetadata(adMetadata)) {
        return INVALID_AD_ACTION;
    }

    return {
        type: ACTION_SET_AD_METADATA,
        currentMetadata: adMetadata,
    };
}

/**
 * Clears the current state of ads in the player
 * to default state of NONE on both contentType and rollType
 *
 * @return {Action}
 */
export function clearCurrentAdMetadata() {
    return setCurrentAdMetadata(NULL_AD_METADATA);
}

export function pauseAd() {
    return function(dispatch, getState) {
        const { adsManager } = getState();
        adsManager.pause();
    };
}

export function playAd() {
    return function(dispatch, getState) {
        const { adsManager } = getState();
        adsManager.play();
    };
}

/**
 * Sets the current known state of adblock for the user.
 *
 * @param {Boolean} detected true if adblock was detected on the user's browser
 * @return {Action}
 */
export function setAdblockDetected(detected) {
    return {
        type: ACTION_SET_ADBLOCK_DETECTED,
        detected,
    };
}

/**
 * Sets the current known vod midroll archive setting for the user/channel.
 *
 * @param {String} setting is off, inserted, or broadcaster based on ad properties API response
 * @return {Action}
 */

export function setVodMidrollArchiveSetting(setting) {
    return {
        type: ACTION_SET_VOD_MIDROLL_ARCHIVE_SETTING,
        setting,
    };
}

/**
 * Sets the current clickthrough URL of the advertisement
 *
 * @param {String} clickthrough URL to overlay on the running ad
 * @return {Action}
 */
export function setAdClickThrough(URL) {
    return {
        type: ACTION_SET_CLICKTHROUGH_URL,
        URL,
    };
}

export function imaScriptLoaded(imaScriptLoaded) {
    return {
        type: ACTION_IMA_SCRIPT_LOADED,
        imaScriptLoaded,
    };
}

/**
 * Gets the request context for a given ads request.
 *
 * @param {String} type Type of ad (PREROLL, MIDROLL, POSTROLL)
 * @param {Number} duration Length of ad break, in seconds
 * @param {Object} state State of the player at the time of the request
 * @param {Boolean} forced Whether ad was forced or not
 * @param {Number} creativeId ad we want to force play
 * @return {Promise<AdsRequestContext>}
 */
export function getRequestContext(rollType, duration, state, forced, creativeId = 0, requestType) {
    return getChannelName(state.stream).then(channelName => {
        return Promise.all([
            userInfo(),
            channelInfo(channelName),
            channelAPIInfo(channelName),
            channelViewerInfo(channelName),
            getChannelAdProperties(channelName),
            getCommunitiesFromChannel(channelName),
            state.experiments.get(PADTIME),
            state.experiments.get(CHOMPERZ),
            state.experiments.get(LEVIATHAN),
        ]);
    }).then(([
        userInfo,
        channelInfo,
        channelAPIInfo,
        viewerInfo,
        channelAdProperties,
        communitiesInfo,
        padtimeExpGroup,
        skippableExpGroup,
        leviathanExpGroup,
    ]) => {
        const { queryKey } = parseUri(state.window.location.href);
        const raid = (queryKey.referrer === 'raid');
        const embedPromo = (queryKey.promo === 'true');
        return new AdsRequestContext({
            adType: rollType,
            sdk: state.sdk,
            lastAdDisplay: localStore.get('lastAdDisplay', 0),
            forced,
            state,
            duration,
            requestType,
            userInfo,
            channelAPIInfo,
            channelInfo,
            viewerInfo,
            channelAdProperties,
            creativeId,
            padtimeExpGroup,
            skippableExpGroup,
            communitiesInfo,
            leviathanExpGroup,
            raid,
            embedPromo,
        });
    });
}

/**
 * Get the channel name associated with a given stream.
 * TODO consolidate with store
 *
 * @param {ContentStream} stream
 * @return {String}
 */
function getChannelName(stream) {
    if (stream instanceof LiveTwitchContentStream) {
        return Promise.resolve(stream.channel);
    } else if (stream instanceof VODTwitchContentStream) {
        return videoInfo(stream.videoId).then(info => info.channel.name);
    }
    return Promise.reject(new Error(`Invalid stream: ${stream}`));
}

/**
 * Check if current ad is a valid one to set in statestore
 *
 * @param {Object} current Ad
 * @return {Boolean}
 */
function isValidAdMetadata(adMetadata) {
    if (!adMetadata.hasOwnProperty('contentType') ||
        !adMetadata.hasOwnProperty('rollType')) {
        return false;
    }

    if (!includes(AdContentTypes, adMetadata.contentType) ||
        !includes(AdRollTypes, adMetadata.rollType)) {
        return false;
    }

    if (adMetadata.contentType === AdContentTypes.NONE &&
        adMetadata.rollType !== AdRollTypes.NONE) {
        return false;
    }

    if (adMetadata.rollType === AdRollTypes.NONE &&
        adMetadata.contentType !== AdContentTypes.NONE) {
        return false;
    }

    return true;
}
