import extend from 'lodash/extend';
import clamp from 'lodash/clamp';
import { DEFAULT_CAPTION } from 'state/captions';
import { fontSizeMap } from 'captions/captions-style-map';
import { localStore } from 'util/storage';

// Local Storage Keys
export const KEY_CAPTIONS_ENABLED = 'captionsEnabled';
export const KEY_CAPTIONS_PRESET = 'captionsPreset';
export const KEY_CAPTIONS_STYLE = 'captionsStyle';

// Actions
export const ACTION_INITIALIZE_CAPTIONS_SETTINGS = 'captions settings initialized';
export const ACTION_SET_CAPTIONS_DATA = 'captions data received';
export const ACTION_SET_CAPTIONS_PRESET = 'captions preset selected';
export const ACTION_TOGGLE_CAPTIONS = 'captions toggled';
export const ACTION_SHOW_CC_MODAL = 'captions show modal';

// Preset Value
export const CUSTOM = 'custom';

/**
 * Opens/Closes the CC options modal
 *
 * @param {Boolean} show - hide/show modal
 * @return {Object}
 */
export function showCCModal(showCCModal) {
    return {
        type: ACTION_SHOW_CC_MODAL,
        showCCModal,
    };
}

/**
 * Set settings for captions
 * @param {Boolean} enableCaptions - Override localStorage if captions are wanted
 * @returns {Object}
 */
export function initializeCaptionsSettings(enableCaptions = false) {
    const enabled = enableCaptions || localStore.get(KEY_CAPTIONS_ENABLED, DEFAULT_CAPTION.enabled);
    return {
        type: ACTION_INITIALIZE_CAPTIONS_SETTINGS,
        captions: {
            enabled,
            preset: localStore.get(KEY_CAPTIONS_PRESET, DEFAULT_CAPTION.preset),
            style: localStore.get(KEY_CAPTIONS_STYLE, DEFAULT_CAPTION.style),
        },
    };
}

/**
 * Set caption data object.
 * @param {Object} captionsData
 * @returns {Object}
 */
export function setCaptionsData(captionsData) {
    return {
        type: ACTION_SET_CAPTIONS_DATA,
        captions: {
            data: captionsData,
        },
    };
}

/**
 * Set whether captions are enabled or disabled.
 * @param bool captionsEnabled
 * @returns {Object}
 */
export function setCaptionsEnabled(captionsEnabled) {
    localStore.set(KEY_CAPTIONS_ENABLED, captionsEnabled);

    return function(dispatch, getState) {
        const captionDisplay = captionsEnabled ? 'block' : 'none';
        const currentStyles = getState().captions.style;
        const validatedStyles = validate(extend({}, currentStyles, { display: captionDisplay }));
        return dispatch({
            type: ACTION_TOGGLE_CAPTIONS,
            captions: {
                enabled: captionsEnabled,
                style: validatedStyles,
            },
        });
    };
}

/**
 * Set which preset to apply to captions.
 * @param String captionsPreset
 * @returns {Object}
 */
export function setCaptionsPreset(presetName, captionStyles = {}) {
    localStore.set(KEY_CAPTIONS_PRESET, presetName);

    return function(dispatch, getState) {
        const currentStyles = getState().captions.style;
        const validatedStyles = validate(extend({}, currentStyles, captionStyles));
        localStore.set(KEY_CAPTIONS_STYLE, validatedStyles);
        return dispatch({
            type: ACTION_SET_CAPTIONS_PRESET,
            captions: {
                preset: presetName,
                style: validatedStyles,
            },
        });
    };
}

function validate(styles) {
    const validSize = clamp(styles.fontSize, fontSizeMap.min, fontSizeMap.max);
    return extend({}, styles, { fontSize: validSize });
}
