import { sessionStore } from '../util/storage';
import { generate } from '../util/unique-id';
import { ONE_HOUR_IN_MS } from '../util/duration-utils';

export const ACTION_REQUEST_COLLECTION = 'request collection';

export const ACTION_SET_COLLECTION = 'set collection';
export const ACTION_CLEAR_COLLECTION = 'clear collection';
export const ACTION_FETCHED_FEATURED_COLLECTION = 'fetched featured collection';

export const ACTION_LOADED_LAST_COLLECTION_ITEM = 'loaded last collection item';
export const ACTION_LOADED_COLLECTION_ITEM = 'loaded collection item';

export const ACTION_OPEN_COLLECTION_SIDEBAR = 'open collection sidebar';
export const ACTION_CLOSE_COLLECTION_SIDEBAR = 'close collection sidebar';

// Session storage key for collection session data
export const COLLECTION_SESSION_KEY = 'collection-session';

// Keys for the object within the collection session object
export const COLLECTION_SESSION_ID_KEY = 'session-id';
export const COLLECTION_SESSION_COLLECTION_ID_KEY = 'collection-id';
export const COLLECTION_SESSION_TIMESTAMP_KEY = 'timestamp';

// Max time (ms) between the last collect MW event and a setCollection that counts as the same collection session
export const COLLECTION_SESSION_TIME_THRESHOLD = ONE_HOUR_IN_MS;

const DEFAULT_THUMBNAIL_URL = 'https://static-cdn.jtvnw.net/ttv-playlists-thumbnails-prod/no-thumbnail-sm.png';

/**
 * Sets Featured Collection Metadata
 * @param {Object} - collection - data from API
 * @return {Object}
 */
export function featuredCollectionFetched(collection) {
    return {
        type: ACTION_FETCHED_FEATURED_COLLECTION,
        featuredCollection: normalizeFeaturedCollectionData(collection),
    };
}

/**
 * Sets collection metadata & video items in that collection.
 * @param {Object} collection object with items filled in
 * @return {Object}
 */
export function setCollection(collection) {
    return function(dispatch) {
        dispatch(setCollectionSessionData(collection._id));
        dispatch({
            type: ACTION_SET_COLLECTION,
            collection: normalizeCollectionData(collection),
        });
    };
}

/**
 * Resets collection id and turns off collection UI and transition overrides
 * @return {Object}
 */
export function clearCollection() {
    return function(dispatch) {
        dispatch(clearCollectionSessionData());
        dispatch({
            type: ACTION_CLEAR_COLLECTION,
        });
    };
}

/**
 * Sets the collection session data in session storage for collection tracking.
 * Collection session data includes:
 * - The collection ID being watched
 * - The last known Unix timestamp since the user was on the session
 * More info: https://jira.twitch.com/browse/VE-122
 *
 * @param {string} collectionId - id of the collection being set
 */
export function setCollectionSessionData(collectionId) {
    return function() {
        const collectionSession = sessionStore.get(COLLECTION_SESSION_KEY, {});
        const lastCollectionSessionId = collectionSession[COLLECTION_SESSION_ID_KEY];
        const lastCollectionId = collectionSession[COLLECTION_SESSION_COLLECTION_ID_KEY];
        const lastCollectionMwTime = collectionSession[COLLECTION_SESSION_TIMESTAMP_KEY];
        const now = Date.now();

        let collectionSessionId;
        const isWithinSessionThreshold = lastCollectionMwTime &&
            lastCollectionId && lastCollectionId === collectionId
            && (now - lastCollectionMwTime) < COLLECTION_SESSION_TIME_THRESHOLD;
        if (isWithinSessionThreshold) {
            collectionSessionId = lastCollectionSessionId;
        } else {
            collectionSessionId = generate();
        }
        sessionStore.set(COLLECTION_SESSION_KEY, {
            [COLLECTION_SESSION_ID_KEY]: collectionSessionId,
            [COLLECTION_SESSION_COLLECTION_ID_KEY]: collectionId,
            [COLLECTION_SESSION_TIMESTAMP_KEY]: now,
        });
    };
}

/**
 * Removes all collection session data from session storage.
 * This should be done whenever a new collection, standalone video, or channel is set.
 *
 * @return {Function} thunk to remove the collection session data
 */
export function clearCollectionSessionData() {
    return function() {
        sessionStore.remove(COLLECTION_SESSION_KEY);
    };
}

/**
 * Fired when the stream for the last item in a collection has been set
 * @return {Object}
 */
export function loadedLastCollectionItem() {
    return {
        type: ACTION_LOADED_LAST_COLLECTION_ITEM,
    };
}

/**
 * Fired when the stream for a collection item from collection transition or selected by the viewer has been set
 * @return {Object}
 */
export function loadedCollectionItem() {
    return {
        type: ACTION_LOADED_COLLECTION_ITEM,
    };
}

/**
 * Sets a pending collection for the collection manager to handle from setPlaylist API
 * @param {string} collectionId - id of the requested collection
 * @param {string} [videoId='']  - id of the requested video to start from
 * @param {boolean} preferVideo - when encountering a mismatch in video and collection,
 *                                ignore collection id and load video if preferVideo is true,
 *                                default to false.
 * @return {Object}
 */
export function requestCollection(collectionId, videoId = '', timestamp = '', preferVideo = false) {
    return {
        type: ACTION_REQUEST_COLLECTION,
        request: {
            collectionId,
            videoId,
            timestamp,
            preferVideo,
        },
    };
}

/**
 * Removes all collection session data from session storage.
 * This should be done whenever a new collection, standalone video, or channel is set.
 *
 * @return {Function} thunk to change the collection session timestamp to the current time
 */
export function updateCollectionSessionTimestamp() {
    return function() {
        const collectionSessionData = sessionStore.get(COLLECTION_SESSION_KEY, {});
        collectionSessionData[COLLECTION_SESSION_TIMESTAMP_KEY] = Date.now();
        sessionStore.set(COLLECTION_SESSION_KEY, collectionSessionData);
    };
}

/**
 * Normalizes featured collection data into a consumable state prop
 *
 * @param {Object} rawCollection - collection data from a given channel
 * @return {Object} normalized response data
 */
function normalizeFeaturedCollectionData(rawCollection) {
    return {
        id: rawCollection._id,
        owner: {
            id: rawCollection.owner._id,
            displayName: rawCollection.owner.display_name,
            name: rawCollection.owner.name,
        },
        thumbnails: rawCollection.thumbnails,
        title: rawCollection.title,
        itemsCount: rawCollection.items_count,
    };
}

/**
 * Normalizes api's collection data into state's collection property
 *
 * @param {Object} api's collection data
 * @return {Object} data in the form of state's collection property
 */
function normalizeCollectionData(rawCollection) {
    return {
        id: rawCollection._id,
        owner: {
            id: rawCollection.owner._id,
            displayName: rawCollection.owner.display_name,
            name: rawCollection.owner.name,
        },
        thumbnails: rawCollection.thumbnails,
        title: rawCollection.title,
        totalDuration: rawCollection.total_duration,
        views: rawCollection.views,
        items: rawCollection.items.map(item => normalizeCollectionItemData(item)),
    };
}

/**
 * Normalizes an item from raw format to normalized format before to set in collection state
 *
 * @param {Object} raw collection item data
 * @return {Object} normalized collection item data to be saved in store
 */
function normalizeCollectionItemData(rawItem) {
    return {
        itemId: rawItem.item_id,
        title: rawItem.title,
        owner: {
            displayName: rawItem.owner.display_name,
        },
        thumbnails: {
            small: rawItem.thumbnails ? rawItem.thumbnails.small : DEFAULT_THUMBNAIL_URL,
        },
    };
}

export function openCollectionSidebar() {
    return {
        type: ACTION_OPEN_COLLECTION_SIDEBAR,
    };
}

export function closeCollectionSidebar() {
    return {
        type: ACTION_CLOSE_COLLECTION_SIDEBAR,
    };
}
