import { ExtensionCoordinator } from 'extension-coordinator';
import { channelInfo, overlayExtensionsForChannel } from '../api';

export const ACTION_SET_EXTENSIONS = 'set extensions';
export const ACTION_CLEAR_EXTENSIONS = 'clear extensions';
export const ACTION_FETCHING_EXTENSIONS = 'fetching extensions';
export const ACTION_EXTENSION_ACTIVATED = 'extension activated';
export const ACTION_REQUEST_TOGGLE_EXTENSION_IDENTITY = 'request toggle extension identity linking';
export const ACTION_SET_EXTENSION_IDENTITY = 'set extension identity linking';
export const ACTION_INIT_EXTENSIONS_API = 'initialize extension api';
export const ACTION_SET_EXTENSION_MODAL = 'set extension modal';
export const ACTION_PURCHASE_INTENT_RECEIVED = 'purchaseIntent received';

export const EMS_REQUEST_PER_SECOND_LIMIT = 3000;
export const EXTENSIONS_NOT_LOADED = 'not loaded';
export const EXTENSIONS_LOADED = 'loaded';
export const EXTENSIONS_LOADING = 'loading';
export const EXTENSIONS_RELOADING = 'reloading';

/**
* Updates extensions visible in the video player.
*
* @param {string} channel channel name the extensions were loaded for
* @param {Array<Object>} extension objects
* @return {Action} action
*/
function setExtensions(channel, extensions) {
    return {
        type: ACTION_SET_EXTENSIONS,
        channel,
        extensions,
    };
}

/**
 * Clears extensions visible in the video player.
 *
 * @returns {Action} action
 */
export function clearExtensions() {
    return {
        type: ACTION_CLEAR_EXTENSIONS,
    };
}

/**
 * Notifies that a network request for available extensions is in progress.
 *
 * @returns {Action} action
 */
function fetchingExtensions(channel) {
    return {
        type: ACTION_FETCHING_EXTENSIONS,
        channel,
    };
}

/**
 * Notifies that we should be reloading the list of extensions from the api
 * because the broadcaster has activated a new extension.
 *
 * @returns {Action} action
 */
function extensionActivated(channel) {
    return {
        type: ACTION_EXTENSION_ACTIVATED,
        channel,
    };
}

/**
 * Requests that a confirmation modal be opened on behalf of coordinator
 * the request contains a callback that will notify coordinator when the
 * modal confirmation is complete and what the result was
 * @param {Object} modalRequest an object which contains:
 *       message: {Object}
 *       - indicates the type of modal that coordinator wishes to open and relevant data needed to contruct it
 *       resultCallback {Function}
 *       - the function to be called with the outcome of the user's interaction with the modal
 * @returns {Action} action
 */
export function setExtensionModal(modalRequest) {
    return {
        type: ACTION_SET_EXTENSION_MODAL,
        modalRequest,
    };
}

export function closeExtensionModal() {
    return {
        type: ACTION_SET_EXTENSION_MODAL,
        modalRequest: null,
    };
}

/**
 * Request that the linked state for a previously loaded extension is toggled.
 * @param {String} extensionId The id of the extension to update.
 */
export function requestExtensionIdentityToggle(extensionId) {
    return {
        type: ACTION_REQUEST_TOGGLE_EXTENSION_IDENTITY,
        extensionId,
    };
}

/**
 * Updates the linked state for a previously loaded extension.
 * @param {String} extensionId The id of the extension to update.
 * @param {Object} isLinked Whether or not the user has linked their identity with the extension.
 */
export function updateExtensionIdentityLinking(extensionId, isLinked) {
    return {
        type: ACTION_SET_EXTENSION_IDENTITY,
        extensionId,
        isLinked,
    };
}

/**
*  Retrieves active extensions targeting the video player for a specific channel.
*  @param {String} channel name of the channel to fetch extensions from.
*  @return {Promise}
*/
export function fetchExtensions(channel) {
    return function(dispatch, getState) {
        const { extensions } = getState();

        if (
            !channel ||
            (
                extensions.channel === channel &&
                extensions.loadingState === EXTENSIONS_LOADING
            )
        ) {
            return Promise.resolve();
        }

        dispatch(fetchingExtensions(channel));

        return overlayExtensionsForChannel(channel).
            then(extensions => {
                const { stream } = getState();
                if (stream.channel !== channel) {
                    return;
                }
                dispatch(setExtensions(channel, extensions));
            });
    };
}

/**
 * Calls subscribeToExtensionControl on the ExtensionService which
 * dispatches setExtensions on activate and removes extensions on deactivate
 *
 * @param {String} channel name of the channel to subscribe to extension-control for
 * @return {Promise}
 */
export function subscribeToExtensionControl(channel) {
    return function(dispatch, getState) {
        const { stream: preflightStream } = getState();
        return channelInfo(channel).
            then(channelInfo => {
                if (getState().stream !== preflightStream) {
                    return;
                }

                const channelId = channelInfo._id;
                ExtensionCoordinator.ExtensionService.subscribeToExtensionControl(channelId, {
                    onShouldFetchExtensions: () => {
                        const { viewercount, window: windowObj } = getState();
                        const jitterOverMilliseconds = (viewercount / EMS_REQUEST_PER_SECOND_LIMIT) * 1000;
                        windowObj.setTimeout(() => {
                            dispatch(extensionActivated(channel));
                            dispatch(fetchExtensions(channel));
                        }, Math.random() * jitterOverMilliseconds);
                    },
                    onDestroyExtension: destroyId => {
                        const notDestroyed = ({ id }) => id !== destroyId;
                        const { activeOverlayExtensions } = getState().extensions;
                        const filteredExtensions = activeOverlayExtensions.filter(notDestroyed);
                        dispatch(setExtensions(channel, filteredExtensions));
                    },
                });
            }).catch(() => Promise.resolve());
    };
}

export function purchaseIntentReceived(purchaseIntent) {
    return {
        type: ACTION_PURCHASE_INTENT_RECEIVED,
        purchaseIntent,
    };
}
