import { getFollowChannel, setFollowChannel, setFollowNotifications } from '../api';

export const ACTION_FOLLOW_INFO_FETCHED = 'follow info fetched';
export const ACTION_PROMPT_LOGIN_MODAL = 'prompt login modal';
export const ACTION_SHOW_FOLLOW_NOTIFICATION = 'show follow notification';

// Tracking Events
export const FOLLOW_AND_NOTIFICATIONS_ON = 'follow_on_notif_on';
export const FOLLOW = 'follow';
export const NOTIFICATIONS_BUTTON_CLICK = 'player_notifbutton_clicks';
export const NOTIFICATIONS_BUTTON_VIEW = 'player_notifbutton_views';
export const NOTIFICATION_CHANGE = 'notification_change';
export const NOTIFICATIONS_OFF = 'notif_off';
export const NOTIFICATIONS_ON = 'notif_on';

/**
 * Sets the boolean for showFollowNotification - which shows the follow panel
 *
 * @param {Boolean} - show
 * @return {Object}
 */
export function showFollowNotification(show) {
    return {
        type: ACTION_SHOW_FOLLOW_NOTIFICATION,
        show,
    };
}

/**
 * This function confirms that the requested follow-info is from the current
 * stream, and not a stream from a previous session
 *
 * @param {Object} oldStream - stream object prior to follow info request
 * @param {Object} currentStream - stream object after follow info request has resolved
 * @return {Boolean}
 */
function isCurrentStream(oldStream, { stream: currentStream }) {
    return oldStream === currentStream;
}

/**
 * Fetches follow info for a user and a given channel
 *
 * @param {Number} userId
 * @param {Number} channelId
 * @return {Function} thunk with getFollowChannel call
 */
export function fetchFollowInfo(userId, channelId) {
    return function(dispatch, getState) {
        const { stream: oldStream } = getState();
        return getFollowChannel(userId, channelId).
            then(followResponse => {
                const notificationsEnabled = followResponse.notifications;
                if (isCurrentStream(oldStream, getState())) {
                    dispatchFollowSuccess(dispatch, followResponse);
                    dispatch(showFollowNotification(!notificationsEnabled));
                }
            }).
            catch(errorMessage => {
                if (isCurrentStream(oldStream, getState())) {
                    dispatchFollowFail(dispatch, errorMessage);
                    dispatch(showFollowNotification(true));
                }
            });
    };
}

/**
 * Adds a user's follow relationship to a given channel
 *
 * @param {Number} userId
 * @param {Number} channelId
 * @return {Function}
 */
export function followChannel(userId, channelId) {
    return function(dispatch) {
        return setFollowChannel(userId, channelId, true).
            then(followResponse => dispatchFollowSuccess(dispatch, followResponse)).
            catch(() => {});
    };
}

/**
 * Removes a user's follow relationship to a given channel
 *
 * @param {Number} userId
 * @param {Number} channelId
 * @return {Function}
 */
export function unfollowChannel(userId, channelId) {
    return function(dispatch) {
        return setFollowChannel(userId, channelId, false).
            then(() => dispatch(followInfoFetched({
                following: false,
                notificationsEnabled: false,
            }))).
            catch(() => {});
    };
}

/**
 * Enables notifications for a given channel
 *
 * @param {Number} userId
 * @param {Number} channelId
 * @return {Function}
 */
export function enableNotifications(userId, channelId) {
    return function(dispatch, getState) {
        const { analyticsTracker } = getState();
        return setFollowNotifications(userId, channelId, true).
            then(followResponse => dispatchFollowSuccess(dispatch, followResponse)).
            then(() => logNotificationsEnabled(analyticsTracker)).
            catch(() => {});
    };
}

/**
 * Disables notifications for a given channel
 *
 * @param {Number} userId
 * @param {Number} channelId
 * @return {Function}
 */
export function disableNotifications(userId, channelId) {
    return function(dispatch, getState) {
        const { analyticsTracker } = getState();
        return setFollowNotifications(userId, channelId, false).
            then(followResponse => dispatchFollowSuccess(dispatch, followResponse)).
            then(() => logNotificationsDisabled(analyticsTracker)).
            catch(() => {});
    };
}

/**
 * Sets the user's follow relationship with a channel
 *
 * @param {Object} followInfo - { following, notificatonsEnabled }
 * @return {Object}
 */
export function followInfoFetched(followInfo = {}) {
    return {
        type: ACTION_FOLLOW_INFO_FETCHED,
        followInfo,
    };
}

/**
 * Updates store on successful calls to follow API
 *
 * @param {Func} dispatch
 * @param {Object} followResponse
 */
export function dispatchFollowSuccess(dispatch, followResponse) {
    const { channel, notifications: notificationsEnabled } = followResponse;

    if (channel._id && (notificationsEnabled !== undefined)) {
        dispatch(followInfoFetched({
            following: true,
            notificationsEnabled,
        }));
    }
}

/**
 * Updates store if follow API 404s with valid `Follow not found` message
 *
 * @param {Func} dispatch
 * @param {Object} errorMessage
 */
export function dispatchFollowFail(dispatch, errorMessage) {
    const { status } = errorMessage;

    if (status === 404) {
        dispatch(followInfoFetched({
            following: false,
            notificationsEnabled: false,
        }));
    }
}

/**
 * Emits an event to web-client to open the login modal
 *
 * @param {String} channelName
 * @return {Object}
 */
export function promptLoginModal(channelName) {
    return {
        type: ACTION_PROMPT_LOGIN_MODAL,
        channelName,
    };
}

/**
 * Sends follow and notification_change events when user succesfully enabled notifications
 *
 * @param {Object} analytics
 */
export function logNotificationsEnabled(analytics) {
    analytics.trackEvent(FOLLOW, { src: 'player' });
    analytics.trackEvent(NOTIFICATION_CHANGE, { notifications: true });
}

/**
 * Sends follow and notification_change events when user succesfully disabled notifications
 *
 * @param {Object} analytics
 */
export function logNotificationsDisabled(analytics) {
    analytics.trackEvent(NOTIFICATION_CHANGE, { notifications: false });
}
